/*
 * Copyright (C) 2019 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// Unit Test for MediaTranscodingService.

//#define LOG_NDEBUG 0
#define LOG_TAG "MediaTranscodingServiceSimulatedTest"

#include <aidl/android/media/BnTranscodingClientCallback.h>
#include <aidl/android/media/IMediaTranscodingService.h>
#include <aidl/android/media/ITranscodingClient.h>
#include <aidl/android/media/ITranscodingClientCallback.h>
#include <aidl/android/media/TranscodingRequestParcel.h>
#include <aidl/android/media/TranscodingSessionParcel.h>
#include <aidl/android/media/TranscodingSessionPriority.h>
#include <android-base/logging.h>
#include <android/binder_manager.h>
#include <android/binder_process.h>
#include <binder/PermissionController.h>
#include <cutils/multiuser.h>
#include <gtest/gtest.h>
#include <utils/Log.h>

#include <iostream>
#include <list>

#include "MediaTranscodingServiceTestHelper.h"
#include "SimulatedTranscoder.h"

namespace android {

namespace media {

// Note that -1 is valid and means using calling pid/uid for the service. But only privilege caller
// could use them. This test is not a privilege caller.
constexpr int32_t kInvalidClientPid = -5;
constexpr int32_t kInvalidClientUid = -10;
constexpr const char* kInvalidClientName = "";
constexpr const char* kInvalidClientOpPackageName = "";

constexpr int64_t kPaddingUs = 1000000;
constexpr int64_t kSessionWithPaddingUs = SimulatedTranscoder::kSessionDurationUs + kPaddingUs;

constexpr const char* kClientOpPackageName = "TestClientPackage";

class MediaTranscodingServiceSimulatedTest : public MediaTranscodingServiceTestBase {
public:
    MediaTranscodingServiceSimulatedTest() { ALOGI("MediaTranscodingServiceResourceTest created"); }

    virtual ~MediaTranscodingServiceSimulatedTest() {
        ALOGI("MediaTranscodingServiceResourceTest destroyed");
    }
};

TEST_F(MediaTranscodingServiceSimulatedTest, TestRegisterNullClient) {
    std::shared_ptr<ITranscodingClient> client;

    // Register the client with null callback.
    Status status = mService->registerClient(nullptr, kClientName, kClientOpPackageName, &client);
    EXPECT_FALSE(status.isOk());
}

TEST_F(MediaTranscodingServiceSimulatedTest, TestRegisterClientWithInvalidClientName) {
    std::shared_ptr<ITranscodingClient> client;

    // Register the client with the service.
    Status status = mService->registerClient(mClient1, kInvalidClientName,
                                             kInvalidClientOpPackageName, &client);
    EXPECT_FALSE(status.isOk());
}

TEST_F(MediaTranscodingServiceSimulatedTest, TestRegisterClientWithInvalidClientPackageName) {
    std::shared_ptr<ITranscodingClient> client;

    // Register the client with the service.
    Status status =
            mService->registerClient(mClient1, kClientName, kInvalidClientOpPackageName, &client);
    EXPECT_FALSE(status.isOk());
}

TEST_F(MediaTranscodingServiceSimulatedTest, TestRegisterOneClient) {
    std::shared_ptr<ITranscodingClient> client;

    Status status = mService->registerClient(mClient1, kClientName, kClientOpPackageName, &client);
    EXPECT_TRUE(status.isOk());

    // Validate the client.
    EXPECT_TRUE(client != nullptr);

    // Check the number of Clients.
    int32_t numOfClients;
    status = mService->getNumOfClients(&numOfClients);
    EXPECT_TRUE(status.isOk());
    EXPECT_GE(numOfClients, 1);

    // Unregister the client.
    status = client->unregister();
    EXPECT_TRUE(status.isOk());
}

TEST_F(MediaTranscodingServiceSimulatedTest, TestRegisterClientTwice) {
    std::shared_ptr<ITranscodingClient> client;

    Status status = mService->registerClient(mClient1, kClientName, kClientOpPackageName, &client);
    EXPECT_TRUE(status.isOk());

    // Validate the client.
    EXPECT_TRUE(client != nullptr);

    // Register the client again and expects failure.
    std::shared_ptr<ITranscodingClient> client1;
    status = mService->registerClient(mClient1, kClientName, kClientOpPackageName, &client1);
    EXPECT_FALSE(status.isOk());

    // Unregister the client.
    status = client->unregister();
    EXPECT_TRUE(status.isOk());
}

TEST_F(MediaTranscodingServiceSimulatedTest, TestRegisterMultipleClients) {
    registerMultipleClients();
    unregisterMultipleClients();
}

TEST_F(MediaTranscodingServiceSimulatedTest, TestSessionIdIndependence) {
    registerMultipleClients();

    // Submit 2 requests on client1 first.
    EXPECT_TRUE(mClient1->submit(0, "test_source_file", "test_destination_file"));
    EXPECT_TRUE(mClient1->submit(1, "test_source_file", "test_destination_file"));

    // Submit 2 requests on client2, sessionId should be independent for each client.
    EXPECT_TRUE(mClient2->submit(0, "test_source_file", "test_destination_file"));
    EXPECT_TRUE(mClient2->submit(1, "test_source_file", "test_destination_file"));

    // Cancel all sessions.
    EXPECT_TRUE(mClient1->cancel(0));
    EXPECT_TRUE(mClient1->cancel(1));
    EXPECT_TRUE(mClient2->cancel(0));
    EXPECT_TRUE(mClient2->cancel(1));

    unregisterMultipleClients();
}

TEST_F(MediaTranscodingServiceSimulatedTest, TestSubmitCancelSessions) {
    registerMultipleClients();

    // Test sessionId assignment.
    EXPECT_TRUE(mClient1->submit(0, "test_source_file_0", "test_destination_file"));
    EXPECT_TRUE(mClient1->submit(1, "test_source_file_1", "test_destination_file"));
    EXPECT_TRUE(mClient1->submit(2, "test_source_file_2", "test_destination_file"));

    // Test submit bad request (no valid sourceFilePath) fails.
    EXPECT_TRUE(mClient1->submit<fail>(0, "", ""));

    // Test submit bad request (no valid sourceFilePath) fails.
    EXPECT_TRUE(mClient1->submit<fail>(0, "src", "dst", TranscodingSessionPriority::kNormal,
                                       1000000, kInvalidClientPid, kInvalidClientUid));

    // Test cancel non-existent session fails.
    EXPECT_TRUE(mClient1->cancel<fail>(100));

    // Session 0 should start immediately and finish in 2 seconds, followed by Session 1 start.
    EXPECT_EQ(mClient1->pop(kPaddingUs), EventTracker::Start(CLIENT(1), 0));
    EXPECT_EQ(mClient1->pop(kSessionWithPaddingUs), EventTracker::Finished(CLIENT(1), 0));
    EXPECT_EQ(mClient1->pop(kPaddingUs), EventTracker::Start(CLIENT(1), 1));

    // Test cancel valid sessionId in random order.
    // Test cancel finished session fails.
    EXPECT_TRUE(mClient1->cancel(2));
    EXPECT_TRUE(mClient1->cancel<fail>(0));
    EXPECT_TRUE(mClient1->cancel(1));

    // Test cancel session again fails.
    EXPECT_TRUE(mClient1->cancel<fail>(1));

    // Test no more events arriving after cancel.
    EXPECT_EQ(mClient1->pop(kSessionWithPaddingUs), EventTracker::NoEvent);

    unregisterMultipleClients();
}

TEST_F(MediaTranscodingServiceSimulatedTest, TestGetSessions) {
    registerMultipleClients();

    // Submit 3 requests.
    EXPECT_TRUE(mClient1->submit(0, "test_source_file_0", "test_destination_file_0"));
    EXPECT_TRUE(mClient1->submit(1, "test_source_file_1", "test_destination_file_1"));
    EXPECT_TRUE(mClient1->submit(2, "test_source_file_2", "test_destination_file_2"));

    // Test get sessions by id.
    EXPECT_TRUE(mClient1->getSession(2, "test_source_file_2", "test_destination_file_2"));
    EXPECT_TRUE(mClient1->getSession(1, "test_source_file_1", "test_destination_file_1"));
    EXPECT_TRUE(mClient1->getSession(0, "test_source_file_0", "test_destination_file_0"));

    // Test get session by invalid id fails.
    EXPECT_TRUE(mClient1->getSession<fail>(100, "", ""));
    EXPECT_TRUE(mClient1->getSession<fail>(-1, "", ""));

    // Test get session after cancel fails.
    EXPECT_TRUE(mClient1->cancel(2));
    EXPECT_TRUE(mClient1->getSession<fail>(2, "", ""));

    // Session 0 should start immediately and finish in 2 seconds, followed by Session 1 start.
    EXPECT_EQ(mClient1->pop(kPaddingUs), EventTracker::Start(CLIENT(1), 0));
    EXPECT_EQ(mClient1->pop(kSessionWithPaddingUs), EventTracker::Finished(CLIENT(1), 0));
    EXPECT_EQ(mClient1->pop(kPaddingUs), EventTracker::Start(CLIENT(1), 1));

    // Test get session after finish fails.
    EXPECT_TRUE(mClient1->getSession<fail>(0, "", ""));

    // Test get the remaining session 1.
    EXPECT_TRUE(mClient1->getSession(1, "test_source_file_1", "test_destination_file_1"));

    // Cancel remaining session 1.
    EXPECT_TRUE(mClient1->cancel(1));

    unregisterMultipleClients();
}

TEST_F(MediaTranscodingServiceSimulatedTest, TestSubmitCancelWithOfflineSessions) {
    registerMultipleClients();

    // Submit some offline sessions first.
    EXPECT_TRUE(mClient1->submit(0, "test_source_file_0", "test_destination_file_0",
                                 TranscodingSessionPriority::kUnspecified));
    EXPECT_TRUE(mClient1->submit(1, "test_source_file_1", "test_destination_file_1",
                                 TranscodingSessionPriority::kUnspecified));

    // Session 0 should start immediately.
    EXPECT_EQ(mClient1->pop(kPaddingUs), EventTracker::Start(CLIENT(1), 0));

    // Submit more real-time sessions.
    EXPECT_TRUE(mClient1->submit(2, "test_source_file_2", "test_destination_file_2"));
    EXPECT_TRUE(mClient1->submit(3, "test_source_file_3", "test_destination_file_3"));

    // Session 0 should pause immediately and session 2 should start.
    EXPECT_EQ(mClient1->pop(kPaddingUs), EventTracker::Pause(CLIENT(1), 0));
    EXPECT_EQ(mClient1->pop(kPaddingUs), EventTracker::Start(CLIENT(1), 2));

    // Session 2 should finish in 2 seconds and session 3 should start.
    EXPECT_EQ(mClient1->pop(kSessionWithPaddingUs), EventTracker::Finished(CLIENT(1), 2));
    EXPECT_EQ(mClient1->pop(kPaddingUs), EventTracker::Start(CLIENT(1), 3));

    // Cancel session 3 now
    EXPECT_TRUE(mClient1->cancel(3));

    // Session 0 should resume and finish in 2 seconds, followed by session 1 start.
    EXPECT_EQ(mClient1->pop(kPaddingUs), EventTracker::Resume(CLIENT(1), 0));
    EXPECT_EQ(mClient1->pop(kSessionWithPaddingUs), EventTracker::Finished(CLIENT(1), 0));
    EXPECT_EQ(mClient1->pop(kPaddingUs), EventTracker::Start(CLIENT(1), 1));

    // Cancel remaining session 1.
    EXPECT_TRUE(mClient1->cancel(1));

    unregisterMultipleClients();
}

TEST_F(MediaTranscodingServiceSimulatedTest, TestClientUseAfterUnregister) {
    std::shared_ptr<ITranscodingClient> client;

    // Register a client, then unregister.
    Status status = mService->registerClient(mClient1, kClientName, kClientOpPackageName, &client);
    EXPECT_TRUE(status.isOk());

    status = client->unregister();
    EXPECT_TRUE(status.isOk());

    // Test various operations on the client, should fail with ERROR_DISCONNECTED.
    TranscodingSessionParcel session;
    bool result;
    status = client->getSessionWithId(0, &session, &result);
    EXPECT_EQ(status.getServiceSpecificError(), IMediaTranscodingService::ERROR_DISCONNECTED);

    status = client->cancelSession(0, &result);
    EXPECT_EQ(status.getServiceSpecificError(), IMediaTranscodingService::ERROR_DISCONNECTED);

    TranscodingRequestParcel request;
    status = client->submitRequest(request, &session, &result);
    EXPECT_EQ(status.getServiceSpecificError(), IMediaTranscodingService::ERROR_DISCONNECTED);
}

TEST_F(MediaTranscodingServiceSimulatedTest, TestTranscodingUidPolicy) {
    ALOGD("TestTranscodingUidPolicy starting...");

    EXPECT_TRUE(ShellHelper::RunCmd("input keyevent KEYCODE_WAKEUP"));
    EXPECT_TRUE(ShellHelper::RunCmd("wm dismiss-keyguard"));
    EXPECT_TRUE(ShellHelper::Stop(kClientPackageA));
    EXPECT_TRUE(ShellHelper::Stop(kClientPackageB));
    EXPECT_TRUE(ShellHelper::Stop(kClientPackageC));

    registerMultipleClients();

    ALOGD("Moving app A to top...");
    EXPECT_TRUE(ShellHelper::Start(kClientPackageA, kTestActivityName));

    // Submit 3 requests.
    ALOGD("Submitting session to client1 (app A) ...");
    EXPECT_TRUE(mClient1->submit(0, "test_source_file_0", "test_destination_file_0"));
    EXPECT_TRUE(mClient1->submit(1, "test_source_file_1", "test_destination_file_1"));
    EXPECT_TRUE(mClient1->submit(2, "test_source_file_2", "test_destination_file_2"));

    // Session 0 should start immediately.
    EXPECT_EQ(mClient1->pop(kPaddingUs), EventTracker::Start(CLIENT(1), 0));

    ALOGD("Moving app B to top...");
    EXPECT_TRUE(ShellHelper::Start(kClientPackageB, kTestActivityName));

    // Session 0 should continue and finish in 2 seconds, then session 1 should start.
    EXPECT_EQ(mClient1->pop(kSessionWithPaddingUs), EventTracker::Finished(CLIENT(1), 0));
    EXPECT_EQ(mClient1->pop(kPaddingUs), EventTracker::Start(CLIENT(1), 1));

    ALOGD("Submitting session to client2 (app B) ...");
    EXPECT_TRUE(mClient2->submit(0, "test_source_file_0", "test_destination_file_0"));

    // Client1's session should pause, client2's session should start.
    EXPECT_EQ(mClient1->pop(kPaddingUs), EventTracker::Pause(CLIENT(1), 1));
    EXPECT_EQ(mClient2->pop(kPaddingUs), EventTracker::Start(CLIENT(2), 0));

    ALOGD("Moving app A back to top...");
    EXPECT_TRUE(ShellHelper::Start(kClientPackageA, kTestActivityName));

    // Client2's session should pause, client1's session 1 should resume.
    EXPECT_EQ(mClient2->pop(kPaddingUs), EventTracker::Pause(CLIENT(2), 0));
    EXPECT_EQ(mClient1->pop(kPaddingUs), EventTracker::Resume(CLIENT(1), 1));

    // Client2's session 1 should finish in 2 seconds, then its session 2 should start.
    EXPECT_EQ(mClient1->pop(kSessionWithPaddingUs), EventTracker::Finished(CLIENT(1), 1));
    EXPECT_EQ(mClient1->pop(kPaddingUs), EventTracker::Start(CLIENT(1), 2));

    // After client2's sessions finish, client1's session should resume.
    EXPECT_EQ(mClient1->pop(kSessionWithPaddingUs), EventTracker::Finished(CLIENT(1), 2));
    EXPECT_EQ(mClient2->pop(kPaddingUs), EventTracker::Resume(CLIENT(2), 0));

    unregisterMultipleClients();

    EXPECT_TRUE(ShellHelper::Stop(kClientPackageA));
    EXPECT_TRUE(ShellHelper::Stop(kClientPackageB));
    EXPECT_TRUE(ShellHelper::Stop(kClientPackageC));

    ALOGD("TestTranscodingUidPolicy finished.");
}

}  // namespace media
}  // namespace android
