/**
 * Copyright (c) 2020, The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.media;

import android.media.TranscodingJobParcel;
import android.media.TranscodingRequestParcel;

/**
 * ITranscodingClient
 *
 * Interface for a client to communicate with MediaTranscodingService.
 *
 * {@hide}
 */
interface ITranscodingClient {
    /**
     * Submits a transcoding request to MediaTranscodingService.
     *
     * @param request a TranscodingRequest contains transcoding configuration.
     * @param job(output variable) a TranscodingJob generated by the MediaTranscodingService.
     * @return true if success, false otherwise.
     */
    boolean submitRequest(in TranscodingRequestParcel request,
                          out TranscodingJobParcel job);

    /**
     * Cancels a transcoding job.
     *
     * @param jobId a TranscodingJob generated by the MediaTranscodingService.
     * @return true if succeeds, false otherwise.
     */
    boolean cancelJob(in int jobId);

    /**
     * Queries the job detail associated with a jobId.
     *
     * @param jobId a TranscodingJob generated by the MediaTranscodingService.
     * @param job(output variable) the TranscodingJob associated with the jobId.
     * @return true if succeeds, false otherwise.
     */
    boolean getJobWithId(in int jobId, out TranscodingJobParcel job);

    /**
    * Unregister the client with the MediaTranscodingService.
    *
    * Client will not be able to perform any more transcoding after unregister.
    */
    void unregister();
}
