/**
 * Copyright (c) 2020, The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.media;

import android.media.TranscodingSessionParcel;
import android.media.TranscodingRequestParcel;

/**
 * ITranscodingClient
 *
 * Interface for a client to communicate with MediaTranscodingService.
 *
 * {@hide}
 */
interface ITranscodingClient {
    /**
     * Submits a transcoding request to MediaTranscodingService.
     *
     * @param request a TranscodingRequest contains transcoding configuration.
     * @param session(output variable) a TranscodingSession generated by MediaTranscodingService.
     * @return true if success, false otherwise.
     */
    boolean submitRequest(in TranscodingRequestParcel request,
                          out TranscodingSessionParcel session);

    /**
     * Cancels a transcoding session.
     *
     * @param sessionId a TranscodingSession generated by the MediaTranscodingService.
     * @return true if succeeds, false otherwise.
     */
    boolean cancelSession(in int sessionId);

    /**
     * Queries the session detail associated with a sessionId.
     *
     * @param sessionId a TranscodingSession generated by the MediaTranscodingService.
     * @param session(output variable) the TranscodingSession associated with the sessionId.
     * @return true if succeeds, false otherwise.
     */
    boolean getSessionWithId(in int sessionId, out TranscodingSessionParcel session);

    /**
    * Unregister the client with the MediaTranscodingService.
    *
    * Client will not be able to perform any more transcoding after unregister.
    */
    void unregister();
}
