/* Copyright (c) 2011, Code Aurora Forum. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */


/*
[SENSOR]
Sensor Model:   OV5640
Camera Module:
Lens Model:
Driver IC:
PV Size         = 640 x 480
Cap Size        = 2592 x 1944
Output Format   = YUYV
MCLK Speed      = 24M
PV DVP_PCLK     = 28M
Cap DVP_PCLK    = 56M
PV Frame Rate   = 30fps
Cap Frame Rate  = 7.5fps
I2C Slave ID    = 0x78
I2C Mode        = 16Addr, 8Data
*/

#ifndef CAMSENSOR_OV5640
#define CAMSENSOR_OV5640

#define INVMASK(v)  (0xff-v)
#define OV5640CORE_WRITEPREG(PTBL)	ov5640_writepregs(PTBL,\
					sizeof(PTBL)/sizeof(PTBL[0]))

/* OV SENSOR SCCB */
struct ov5640_sensor {
	uint16_t addr;
	uint8_t data;
	uint8_t mask;
};

/* Auto Focus Command */
#define OV5640_CMD_MAIN 0x3022
#define OV5640_CMD_ACK 0x3023
#define OV5640_CMD_PARA0 0x3024
#define OV5640_CMD_PARA1 0x3025
#define OV5640_CMD_PARA2 0x3026
#define OV5640_CMD_PARA3 0x3027
#define OV5640_CMD_PARA4 0x3028
#define OV5640_CMD_FW_STATUS 0x3029

/* Sensor ID */
#define OV5640_SENSOR_ID 0x5640

#define capture_framerate 750     /* 7.5fps capture frame rate */
#define g_preview_frameRate 3000  /* 30fps preview frame rate */

struct ov5640_sensor ov5640_init_tbl[] = {
	{0x3008, 0x42},
	{0x3103, 0x03},
	{0x3017, 0x00},
	{0x3018, 0x00},
	{0x3034, 0x18},
	{0x3035, 0x14},
	{0x3036, 0x38},
	{0x3037, 0x13},
	{0x3108, 0x01},
	{0x3630, 0x36},
	{0x3631, 0x0e},
	{0x3632, 0xe2},
	{0x3633, 0x12},
	{0x3621, 0xe0},
	{0x3704, 0xa0},
	{0x3703, 0x5a},
	{0x3715, 0x78},
	{0x3717, 0x01},
	{0x370b, 0x60},
	{0x3705, 0x1a},
	{0x3905, 0x02},
	{0x3906, 0x10},
	{0x3901, 0x0a},
	{0x3731, 0x12},
	{0x3600, 0x08},
	{0x3601, 0x33},
	{0x302d, 0x60},
	{0x3620, 0x52},
	{0x371b, 0x20},
	{0x471c, 0x50},
	{0x3a13, 0x43},
	{0x3a18, 0x00},
	{0x3a19, 0xf8},
	{0x3635, 0x13},
	{0x3636, 0x03},
	{0x3634, 0x40},
	{0x3622, 0x01},
	{0x3c01, 0x34},
	{0x3c04, 0x28},
	{0x3c05, 0x98},
	{0x3c06, 0x00},
	{0x3c07, 0x08},
	{0x3c08, 0x00},
	{0x3c09, 0x1c},
	{0x3c0a, 0x9c},
	{0x3c0b, 0x40},
	{0x3820, 0x41},
	{0x3821, 0x07},
	{0x3814, 0x31},
	{0x3815, 0x31},
	{0x3800, 0x00},
	{0x3801, 0x00},
	{0x3802, 0x00},
	{0x3803, 0x04},
	{0x3804, 0x0a},
	{0x3805, 0x3f},
	{0x3806, 0x07},
	{0x3807, 0x9b},
	{0x3808, 0x02},
	{0x3809, 0x80},
	{0x380a, 0x01},
	{0x380b, 0xe0},
	{0x380c, 0x07},
	{0x380d, 0x68},
	{0x380e, 0x03},
	{0x380f, 0xd8},
	{0x3810, 0x00},
	{0x3811, 0x10},
	{0x3812, 0x00},
	{0x3813, 0x06},
	{0x3618, 0x00},
	{0x3612, 0x29},
	{0x3708, 0x64},
	{0x3709, 0x52},
	{0x370c, 0x03},
	{0x3a02, 0x03},
	{0x3a03, 0xd8},
	{0x3a08, 0x01},
	{0x3a09, 0x27},
	{0x3a0a, 0x00},
	{0x3a0b, 0xf6},
	{0x3a0e, 0x03},
	{0x3a0d, 0x04},
	{0x3a14, 0x03},
	{0x3a15, 0xd8},
	{0x4001, 0x02},
	{0x4004, 0x02},
	{0x3000, 0x00},
	{0x3002, 0x1c},
	{0x3004, 0xff},
	{0x3006, 0xc3},
	{0x300e, 0x45},
	{0x302e, 0x08},
	{0x4300, 0x30},
	{0x501f, 0x00},
	{0x4713, 0x03},
	{0x4407, 0x04},
	{0x440e, 0x00},
	{0x460b, 0x35},
	{0x460c, 0x22},
	{0x4837, 0x44},
	{0x3824, 0x02},
	{0x5000, 0xa7},
	{0x5001, 0xa3},
	{0x5180, 0xff},
	{0x5181, 0xf2},
	{0x5182, 0x00},
	{0x5183, 0x14},
	{0x5184, 0x25},
	{0x5185, 0x24},
	{0x5186, 0x09},
	{0x5187, 0x09},
	{0x5188, 0x09},
	{0x5189, 0x75},
	{0x518a, 0x54},
	{0x518b, 0xe0},
	{0x518c, 0xb2},
	{0x518d, 0x42},
	{0x518e, 0x3d},
	{0x518f, 0x56},
	{0x5190, 0x46},
	{0x5191, 0xf8},
	{0x5192, 0x04},
	{0x5193, 0x70},
	{0x5194, 0xf0},
	{0x5195, 0xf0},
	{0x5196, 0x03},
	{0x5197, 0x01},
	{0x5198, 0x04},
	{0x5199, 0x12},
	{0x519a, 0x04},
	{0x519b, 0x00},
	{0x519c, 0x06},
	{0x519d, 0x82},
	{0x519e, 0x38},
	{0x5381, 0x1e},
	{0x5382, 0x5b},
	{0x5383, 0x08},
	{0x5384, 0x0a},
	{0x5385, 0x7e},
	{0x5386, 0x88},
	{0x5387, 0x7c},
	{0x5388, 0x6c},
	{0x5389, 0x10},
	{0x538a, 0x01},
	{0x538b, 0x98},
	{0x5300, 0x08},
	{0x5301, 0x30},
	{0x5302, 0x10},
	{0x5303, 0x00},
	{0x5304, 0x08},
	{0x5305, 0x30},
	{0x5306, 0x08},
	{0x5307, 0x16},
	{0x5309, 0x08},
	{0x530a, 0x30},
	{0x530b, 0x04},
	{0x530c, 0x06},
	{0x5480, 0x01},
	{0x5481, 0x08},
	{0x5482, 0x14},
	{0x5483, 0x28},
	{0x5484, 0x51},
	{0x5485, 0x65},
	{0x5486, 0x71},
	{0x5487, 0x7d},
	{0x5488, 0x87},
	{0x5489, 0x91},
	{0x548a, 0x9a},
	{0x548b, 0xaa},
	{0x548c, 0xb8},
	{0x548d, 0xcd},
	{0x548e, 0xdd},
	{0x548f, 0xea},
	{0x5490, 0x1d},
	{0x5580, 0x02},
	{0x5583, 0x40},
	{0x5584, 0x10},
	{0x5589, 0x10},
	{0x558a, 0x00},
	{0x558b, 0xf8},
	{0x5800, 0x23},
	{0x5801, 0x14},
	{0x5802, 0x0f},
	{0x5803, 0x0f},
	{0x5804, 0x12},
	{0x5805, 0x26},
	{0x5806, 0x0c},
	{0x5807, 0x08},
	{0x5808, 0x05},
	{0x5809, 0x05},
	{0x580a, 0x08},
	{0x580b, 0x0d},
	{0x580c, 0x08},
	{0x580d, 0x03},
	{0x580e, 0x00},
	{0x580f, 0x00},
	{0x5810, 0x03},
	{0x5811, 0x09},
	{0x5812, 0x07},
	{0x5813, 0x03},
	{0x5814, 0x00},
	{0x5815, 0x01},
	{0x5816, 0x03},
	{0x5817, 0x08},
	{0x5818, 0x0d},
	{0x5819, 0x08},
	{0x581a, 0x05},
	{0x581b, 0x06},
	{0x581c, 0x08},
	{0x581d, 0x0e},
	{0x581e, 0x29},
	{0x581f, 0x17},
	{0x5820, 0x11},
	{0x5821, 0x11},
	{0x5822, 0x15},
	{0x5823, 0x28},
	{0x5824, 0x46},
	{0x5825, 0x26},
	{0x5826, 0x08},
	{0x5827, 0x26},
	{0x5828, 0x64},
	{0x5829, 0x26},
	{0x582a, 0x24},
	{0x582b, 0x22},
	{0x582c, 0x24},
	{0x582d, 0x24},
	{0x582e, 0x06},
	{0x582f, 0x22},
	{0x5830, 0x40},
	{0x5831, 0x42},
	{0x5832, 0x24},
	{0x5833, 0x26},
	{0x5834, 0x24},
	{0x5835, 0x22},
	{0x5836, 0x22},
	{0x5837, 0x26},
	{0x5838, 0x44},
	{0x5839, 0x24},
	{0x583a, 0x26},
	{0x583b, 0x28},
	{0x583c, 0x42},
	{0x583d, 0xce},
	{0x5025, 0x00},
	{0x3a0f, 0x30},
	{0x3a10, 0x28},
	{0x3a1b, 0x30},
	{0x3a1e, 0x26},
	{0x3a11, 0x60},
	{0x3a1f, 0x14},
	{0x3008, 0x02},
};

struct ov5640_sensor ov5640_init_iq_tbl[] = {
/* Lens correction */
/* OV5640 LENC setting */
	{0x5800, 0x3f},
	{0x5801, 0x20},
	{0x5802, 0x1a},
	{0x5803, 0x1a},
	{0x5804, 0x23},
	{0x5805, 0x3f},
	{0x5806, 0x11},
	{0x5807, 0x0c},
	{0x5808, 0x09},
	{0x5809, 0x08},
	{0x580a, 0x0d},
	{0x580b, 0x12},
	{0x580c, 0x0d},
	{0x580d, 0x04},
	{0x580e, 0x00},
	{0x580f, 0x00},
	{0x5810, 0x05},
	{0x5811, 0x0d},
	{0x5812, 0x0d},
	{0x5813, 0x04},
	{0x5814, 0x00},
	{0x5815, 0x00},
	{0x5816, 0x04},
	{0x5817, 0x0d},
	{0x5818, 0x13},
	{0x5819, 0x0d},
	{0x581a, 0x08},
	{0x581b, 0x08},
	{0x581c, 0x0c},
	{0x581d, 0x13},
	{0x581e, 0x3f},
	{0x581f, 0x1f},
	{0x5820, 0x1b},
	{0x5821, 0x1c},
	{0x5822, 0x23},
	{0x5823, 0x3f},
	{0x5824, 0x6a},
	{0x5825, 0x06},
	{0x5826, 0x08},
	{0x5827, 0x06},
	{0x5828, 0x2a},
	{0x5829, 0x08},
	{0x582a, 0x24},
	{0x582b, 0x24},
	{0x582c, 0x24},
	{0x582d, 0x08},
	{0x582e, 0x08},
	{0x582f, 0x22},
	{0x5830, 0x40},
	{0x5831, 0x22},
	{0x5832, 0x06},
	{0x5833, 0x08},
	{0x5834, 0x24},
	{0x5835, 0x24},
	{0x5836, 0x04},
	{0x5837, 0x0a},
	{0x5838, 0x86},
	{0x5839, 0x08},
	{0x583a, 0x28},
	{0x583b, 0x28},
	{0x583c, 0x66},
	{0x583d, 0xce},
/* AEC */
	{0x3a0f, 0x38},
	{0x3a10, 0x30},
	{0x3a11, 0x61},
	{0x3a1b, 0x38},
	{0x3a1e, 0x30},
	{0x3a1f, 0x10},
	/* AWB */
	{0x5180, 0xff},
	{0x5181, 0xf2},
	{0x5182, 0x00},
	{0x5183, 0x14},
	{0x5184, 0x25},
	{0x5185, 0x24},
	{0x5186, 0x09},
	{0x5187, 0x09},
	{0x5188, 0x09},
	{0x5189, 0x88},
	{0x518a, 0x54},
	{0x518b, 0xee},
	{0x518c, 0xb2},
	{0x518d, 0x50},
	{0x518e, 0x34},
	{0x518f, 0x6b},
	{0x5190, 0x46},
	{0x5191, 0xf8},
	{0x5192, 0x04},
	{0x5193, 0x70},
	{0x5194, 0xf0},
	{0x5195, 0xf0},
	{0x5196, 0x03},
	{0x5197, 0x01},
	{0x5198, 0x04},
	{0x5199, 0x6c},
	{0x519a, 0x04},
	{0x519b, 0x00},
	{0x519c, 0x09},
	{0x519d, 0x2b},
	{0x519e, 0x38},

/* UV Adjust Auto Mode */
	{0x5580, 0x02},	/* 02 ;Sat enable */
	{0x5588, 0x01},	/*40 ;enable UV adj */
	{0x5583, 0x40},	/*	;offset high */
	{0x5584, 0x18},	/*	;offset low */
	{0x5589, 0x18},	/*	;gth1	*/
	{0x558a, 0x00},
	{0x358b, 0xf8},	/*	;gth2 */
};

struct ov5640_sensor ov5640_preview_tbl[] = {
/* @@ MIPI_2lane_5M to vga(YUV) 30fps 99 640 480 98 0 0 */
	{0x3503, 0x00}, /* enable AE back from capture to preview */
	{0x3035, 0x14},
	{0x3036, 0x38},
	{0x3820, 0x41},
	{0x3821, 0x07},
	{0x3814, 0x31},
	{0x3815, 0x31},
	{0x3803, 0x04},
	{0x3807, 0x9b},
	{0x3808, 0x02},
	{0x3809, 0x80},
	{0x380a, 0x01},
	{0x380b, 0xe0},
	{0x380c, 0x07},
	{0x380d, 0x68},
	{0x380e, 0x03},
	{0x380f, 0xd8},
	{0x3813, 0x06},
	{0x3618, 0x00},
	{0x3612, 0x29},
	{0x3708, 0x64},
	{0x3709, 0x52},
	{0x370c, 0x03},
	{0x5001, 0xa3},
	{0x4004, 0x02},
	{0x4005, 0x18},
	{0x4837, 0x44},
	{0x4713, 0x03},
	{0x4407, 0x04},
	{0x460b, 0x35},
	{0x460c, 0x22},
	{0x3824, 0x02},
};

struct ov5640_sensor ov5640_capture_tbl[] = {
/* @@ MIPI_2lane_5M(YUV) 7.5/15fps 99 2592 1944 98 0 0 */
	{0x3035, 0x21}, /* 11 */
	{0x3036, 0x54},
	{0x3820, 0x40},
	{0x3821, 0x06},
	{0x3814, 0x11},
	{0x3815, 0x11},
	{0x3803, 0x00},
	{0x3807, 0x9f},
	{0x3808, 0x0a},
	{0x3809, 0x20},
	{0x380a, 0x07},
	{0x380b, 0x98},
	{0x380c, 0x0b},
	{0x380d, 0x1c},
	{0x380e, 0x07},
	{0x380f, 0xb0},
	{0x3813, 0x04},
	{0x3618, 0x04},
	{0x3612, 0x2b},
	{0x3708, 0x21},
	{0x3709, 0x12},
	{0x370c, 0x00},
	{0x5001, 0x83},
	{0x4004, 0x06},
	{0x4005, 0x1a},
	{0x4837, 0x15}, /* 0a */
	{0x4713, 0x02},
	{0x4407, 0x0c},
	{0x460b, 0x37},
	{0x460c, 0x20},
	{0x3824, 0x01},
};

/* Contrast */

struct ov5640_sensor ov5640_contrast_lv0_tbl[] = {
/* Contrast -4 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5580, 0x04, INVMASK(0x04)}, /* Enable BIT2 for contrast/brightness
					  control*/
	{0x5586, 0x10},                /* Gain */
	{0x5585, 0x10},                /* Offset */
	{0x5588, 0x00, INVMASK(0x04)}, /* Offset sign */
};

struct ov5640_sensor ov5640_contrast_lv1_tbl[] = {
/* Contrast -3 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5580, 0x04, INVMASK(0x04)}, /* Enable BIT2 for contrast/brightness
					  control */
	{0x5586, 0x14},                /* Gain */
	{0x5585, 0x14},                /* Offset */
	{0x5588, 0x00, INVMASK(0x04)}, /* Offset sign */
};

struct ov5640_sensor ov5640_contrast_lv2_tbl[] = {
/* Contrast -2 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5580, 0x04, INVMASK(0x04)}, /* Enable BIT2 for contrast/brightness
					  control */
	{0x5586, 0x18},                /* Gain */
	{0x5585, 0x18},                /* Offset */
	{0x5588, 0x00, INVMASK(0x04)}, /* Offset sign */
};

struct ov5640_sensor ov5640_contrast_lv3_tbl[] = {
/* Contrast -1 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5580, 0x04, INVMASK(0x04)},
	{0x5586, 0x1c},
	{0x5585, 0x1c},
	{0x5588, 0x00, INVMASK(0x04)},
};

struct ov5640_sensor ov5640_contrast_default_lv4_tbl[] = {
/* Contrast (Default) */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5580, 0x04, INVMASK(0x04)},
	{0x5586, 0x20},
	{0x5585, 0x00},
	{0x5588, 0x00, INVMASK(0x04)},
};

struct ov5640_sensor ov5640_contrast_lv5_tbl[] = {
/* Contrast +1 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5580, 0x04, INVMASK(0x04)},
	{0x5586, 0x24},
	{0x5585, 0x10},
	{0x5588, 0x00, INVMASK(0x04)},
};

struct ov5640_sensor ov5640_contrast_lv6_tbl[] = {
/* Contrast +2 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5580, 0x04, INVMASK(0x04)},
	{0x5586, 0x28},
	{0x5585, 0x18},
	{0x5588, 0x00, INVMASK(0x04)},
};

struct ov5640_sensor ov5640_contrast_lv7_tbl[] = {
/* Contrast +3 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5580, 0x04, INVMASK(0x04)},
	{0x5586, 0x2c},
	{0x5585, 0x1c},
	{0x5588, 0x00, INVMASK(0x04)},
};

struct ov5640_sensor ov5640_contrast_lv8_tbl[] = {
/* Contrast +4 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5580, 0x04, INVMASK(0x04)},
	{0x5586, 0x30},
	{0x5585, 0x20},
	{0x5588, 0x00, INVMASK(0x04)},
};

/* Sharpness */

struct ov5640_sensor ov5640_sharpness_lv0_tbl[] = {
/* Sharpness 0 */
	{0x5308, 0x40, INVMASK(0x40)},
	{0x5302, 0x00},
};

struct ov5640_sensor ov5640_sharpness_lv1_tbl[] = {
/* Sharpness 1 */
	{0x5308, 0x40, INVMASK(0x40)},
	{0x5302, 0x02},
};

struct ov5640_sensor ov5640_sharpness_default_lv2_tbl[] = {
/* Sharpness_Auto (Default) */
	{0x5308, 0x00, INVMASK(0x40)},
	{0x5300, 0x08},
	{0x5301, 0x30},
	{0x5302, 0x10},
	{0x5303, 0x00},
	{0x5309, 0x08},
	{0x530a, 0x30},
	{0x530b, 0x04},
	{0x530c, 0x06},
};

struct ov5640_sensor ov5640_sharpness_lv3_tbl[] = {
/* Sharpness 3 */
	{0x5308, 0x40, INVMASK(0x40)},
	{0x5302, 0x08},
};

struct ov5640_sensor ov5640_sharpness_lv4_tbl[] = {
/* Sharpness 4 */
	{0x5308, 0x40, INVMASK(0x40)},
	{0x5302, 0x0c},
};

struct ov5640_sensor ov5640_sharpness_lv5_tbl[] = {
/* Sharpness 5 */
	{0x5308, 0x40, INVMASK(0x40)},
	{0x5302, 0x10},
};

struct ov5640_sensor ov5640_sharpness_lv6_tbl[] = {
/* Sharpness 6 */
	{0x5308, 0x40, INVMASK(0x40)},
	{0x5302, 0x14},
};

struct ov5640_sensor ov5640_sharpness_lv7_tbl[] = {
/* Sharpness 7 */
	{0x5308, 0x40, INVMASK(0x40)},
	{0x5302, 0x18},
};

struct ov5640_sensor ov5640_sharpness_lv8_tbl[] = {
/* Sharpness 8 */
	{0x5308, 0x40, INVMASK(0x40)},
	{0x5302, 0x20},
};

/* Saturation */

struct ov5640_sensor ov5640_saturation_lv0_tbl[] = {
/* Saturation x0.25 */
	{0x5001, 0x83, INVMASK(0x80)},  /* SDE_En */
	{0x5583, 0x00},                 /* Saturaion gain in U */
	{0x5584, 0x00},                 /* Saturation gain in V */
	{0x5580, 0x02, INVMASK(0x02)},  /* Saturation enable */
	{0x5588, 0x40, INVMASK(0x40)},
};

struct ov5640_sensor ov5640_saturation_lv1_tbl[] = {
/* Saturation x0.5 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5583, 0x10},
	{0x5584, 0x10},
	{0x5580, 0x02, INVMASK(0x02)},
	{0x5588, 0x40, INVMASK(0x40)},
};

struct ov5640_sensor ov5640_saturation_lv2_tbl[] = {
/* Saturation x0.75 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5583, 0x20},
	{0x5584, 0x20},
	{0x5580, 0x02, INVMASK(0x02)},
	{0x5588, 0x40, INVMASK(0x40)},
};

struct ov5640_sensor ov5640_saturation_lv3_tbl[] = {
/* Saturation x0.75 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5583, 0x30},
	{0x5584, 0x30},
	{0x5580, 0x02, INVMASK(0x02)},
	{0x5588, 0x40, INVMASK(0x40)},
};

struct ov5640_sensor ov5640_saturation_default_lv4_tbl[] = {
/* Saturation x1 (Default) */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5583, 0x40},
	{0x5584, 0x40},
	{0x5580, 0x02, INVMASK(0x02)},
	{0x5588, 0x40, INVMASK(0x40)},
};

struct ov5640_sensor ov5640_saturation_lv5_tbl[] = {
/* Saturation x1.25 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5583, 0x50},
	{0x5584, 0x50},
	{0x5580, 0x02, INVMASK(0x02)},
	{0x5588, 0x40, INVMASK(0x40)},
};

struct ov5640_sensor ov5640_saturation_lv6_tbl[] = {
/* Saturation x1.5 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5583, 0x60},
	{0x5584, 0x60},
	{0x5580, 0x02, INVMASK(0x02)},
	{0x5588, 0x40, INVMASK(0x40)},
};

struct ov5640_sensor ov5640_saturation_lv7_tbl[] = {
/* Saturation x1.25 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5583, 0x70},
	{0x5584, 0x70},
	{0x5580, 0x02, INVMASK(0x02)},
	{0x5588, 0x40, INVMASK(0x40)},
};

struct ov5640_sensor ov5640_saturation_lv8_tbl[] = {
/* Saturation x1.5 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5583, 0x80},
	{0x5584, 0x80},
	{0x5580, 0x02, INVMASK(0x02)},
	{0x5588, 0x40, INVMASK(0x40)},
};

/* Brightness */

struct ov5640_sensor ov5640_brightness_lv0_tbl[] = {
/* Brightness -4 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5587, 0x40},
	{0x5580, 0x04, INVMASK(0x04)},
	{0x5588, 0x08, INVMASK(0x08)},
};

struct ov5640_sensor ov5640_brightness_lv1_tbl[] = {
/* Brightness -3 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5587, 0x30},
	{0x5580, 0x04, INVMASK(0x04)},
	{0x5588, 0x08, INVMASK(0x08)},
};

struct ov5640_sensor ov5640_brightness_lv2_tbl[] = {
/* Brightness -2 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5587, 0x20},
	{0x5580, 0x04, INVMASK(0x04)},
	{0x5588, 0x08, INVMASK(0x08)},
};

struct ov5640_sensor ov5640_brightness_lv3_tbl[] = {
/* Brightness -1 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5587, 0x10},
	{0x5580, 0x04, INVMASK(0x04)},
	{0x5588, 0x08, INVMASK(0x08)},
};

struct ov5640_sensor ov5640_brightness_default_lv4_tbl[] = {
/* Brightness 0 (Default) */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5587, 0x00},
	{0x5580, 0x04, INVMASK(0x04)},
	{0x5588, 0x00, INVMASK(0x08)},
};

struct ov5640_sensor ov5640_brightness_lv5_tbl[] = {
/* Brightness +1 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5587, 0x10},
	{0x5580, 0x04, INVMASK(0x04)},
	{0x5588, 0x00, INVMASK(0x08)},
};

struct ov5640_sensor ov5640_brightness_lv6_tbl[] = {
/* Brightness +2 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5587, 0x20},
	{0x5580, 0x04, INVMASK(0x04)},
	{0x5588, 0x00, INVMASK(0x08)},
};

struct ov5640_sensor ov5640_brightness_lv7_tbl[] = {
/* Brightness +3 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5587, 0x30},
	{0x5580, 0x04, INVMASK(0x04)},
	{0x5588, 0x00, INVMASK(0x08)},
};

struct ov5640_sensor ov5640_brightness_lv8_tbl[] = {
/* Brightness +4 */
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5587, 0x40},
	{0x5580, 0x04, INVMASK(0x04)},
	{0x5588, 0x00, INVMASK(0x08)},
};

/* Exposure Compensation */
struct ov5640_sensor ov5640_exposure_compensation_lv0_tbl[] = {
	/* @@ +1.7EV */
	{0x3a0f, 0x60},
	{0x3a10, 0x58},
	{0x3a11, 0xa0},
	{0x3a1b, 0x60},
	{0x3a1e, 0x58},
	{0x3a1f, 0x20},
};

struct ov5640_sensor ov5640_exposure_compensation_lv1_tbl[] = {
	/* @@ +1.0EV */
	{0x3a0f, 0x50},
	{0x3a10, 0x48},
	{0x3a11, 0x90},
	{0x3a1b, 0x50},
	{0x3a1e, 0x48},
	{0x3a1f, 0x20},
};

struct ov5640_sensor ov5640_exposure_compensation_lv2_default_tbl[] = {
	/* @@ default */
	{0x3a0f, 0x38},
	{0x3a10, 0x30},
	{0x3a11, 0x61},
	{0x3a1b, 0x38},
	{0x3a1e, 0x30},
	{0x3a1f, 0x10},
};

struct ov5640_sensor ov5640_exposure_compensation_lv3_tbl[] = {
	/* @@ -1.0EV */
	{0x3a0f, 0x20},
	{0x3a10, 0x18},
	{0x3a11, 0x41},
	{0x3a1b, 0x20},
	{0x3a1e, 0x18},
	{0x3a1f, 0x10},
};

struct ov5640_sensor ov5640_exposure_compensation_lv4_tbl[] = {
	/* @@ -1.7EV */
	{0x3a0f, 0x10},
	{0x3a10, 0x08},
	{0x3a11, 0x10},
	{0x3a1b, 0x08},
	{0x3a1e, 0x20},
	{0x3a1f, 0x10},
};

/* Auto Expourse Weight */

struct ov5640_sensor ov5640_ae_average_tbl[] = {
  /* Whole Image Average */
	{0x5688, 0x11}, /* Zone 1/Zone 0 weight */
	{0x5689, 0x11}, /* Zone 3/Zone 2 weight */
	{0x569a, 0x11}, /* Zone 5/Zone 4 weight */
	{0x569b, 0x11}, /* Zone 7/Zone 6 weight */
	{0x569c, 0x11}, /* Zone 9/Zone 8 weight */
	{0x569d, 0x11}, /* Zone b/Zone a weight */
	{0x569e, 0x11}, /* Zone d/Zone c weight */
	{0x569f, 0x11}, /* Zone f/Zone e weight */
};

struct ov5640_sensor ov5640_ae_centerweight_tbl[] = {
  /* Whole Image Center More weight */
	{0x5688, 0x62},
	{0x5689, 0x26},
	{0x568a, 0xe6},
	{0x568b, 0x6e},
	{0x568c, 0xea},
	{0x568d, 0xae},
	{0x568e, 0xa6},
	{0x568f, 0x6a},
};

/* Light Mode */
struct ov5640_sensor ov5640_wb_def[] = {
	{0x3406, 0x00, INVMASK(0x01)},
};

struct ov5640_sensor ov5640_wb_custom[] = {
	{0x3406, 0x01, INVMASK(0x01)},
	{0x3400, 0x04},
	{0x3401, 0x58},
	{0x3402, 0x04},
	{0x3403, 0x00},
	{0x3404, 0x08},
	{0x3405, 0x40},
};

struct ov5640_sensor ov5640_wb_inc[] = {
	{0x3406, 0x01, INVMASK(0x01)},
	{0x3400, 0x04},
	{0x3401, 0x88},
	{0x3402, 0x04},
	{0x3403, 0x00},
	{0x3404, 0x08},
	{0x3405, 0xb6},
};

struct ov5640_sensor ov5640_wb_daylight[] = {
	{0x3406, 0x01, INVMASK(0x01)},
	{0x3400, 0x07},
	{0x3401, 0x02},
	{0x3402, 0x04},
	{0x3403, 0x00},
	{0x3404, 0x05},
	{0x3405, 0x15},
};

struct ov5640_sensor ov5640_wb_cloudy[] = {
	{0x3406, 0x01, INVMASK(0x01)},
	{0x3400, 0x07},
	{0x3401, 0x88},
	{0x3402, 0x04},
	{0x3403, 0x00},
	{0x3404, 0x05},
	{0x3405, 0x00},
};

/* EFFECT */
struct ov5640_sensor ov5640_effect_normal_tbl[] = {
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5580, 0x00, INVMASK(0x78)},
	{0x5003, 0x08},
	{0x5583, 0x40},
	{0x5584, 0x40},
};

struct ov5640_sensor ov5640_effect_mono_tbl[] = {
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5580, 0x20, INVMASK(0x78)},
	{0x5003, 0x08},
	{0x5583, 0x40},
	{0x5584, 0x40},
};

struct ov5640_sensor ov5640_effect_bw_tbl[] = {
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5580, 0x18, INVMASK(0x78)},
	{0x5003, 0x08},
	{0x5583, 0x80},
	{0x5584, 0x80},
};

struct ov5640_sensor ov5640_effect_bluish_tbl[] = {
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5580, 0x18, INVMASK(0x78)},
	{0x5003, 0x08},
	{0x5583, 0xa0},
	{0x5584, 0x40},
};

struct ov5640_sensor ov5640_effect_solarize_tbl[] = {
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5580, 0x00, INVMASK(0x78)},
	{0x5003, 0x09},
};


struct ov5640_sensor ov5640_effect_sepia_tbl[] = {
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5580, 0x18, INVMASK(0x78)},
	{0x5003, 0x08},
	{0x5583, 0x40},
	{0x5584, 0xa0},
};

struct ov5640_sensor ov5640_effect_reddish_tbl[] = {
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5580, 0x18, INVMASK(0x78)},
	{0x5003, 0x08},
	{0x5583, 0x80},
	{0x5584, 0xc0},
};

struct ov5640_sensor ov5640_effect_greenish_tbl[] = {
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5580, 0x18, INVMASK(0x78)},
	{0x5003, 0x08},
	{0x5583, 0x60},
	{0x5584, 0x60},
};

struct ov5640_sensor ov5640_effect_negative_tbl[] = {
	{0x5001, 0x83, INVMASK(0x80)},
	{0x5580, 0x40, INVMASK(0x78)},
	{0x5003, 0x08},
};

/* AntiBanding */
struct ov5640_sensor ov5640_antibanding_auto_tbl[] = {
  /* Auto-XCLK24MHz */
	{0x3622, 0x01}, /* PD-sel */
	{0x3635, 0x1c}, /* VMREF 3635[2:0] */
	{0x3634, 0x40}, /* I_5060 3643[2:0] */
	{0x3c01, 0x34},
	{0x3c00, 0x00},
	{0x3c04, 0x28},
	{0x3c05, 0x98},
	{0x3c06, 0x00},
	{0x3c07, 0x08},
	{0x3c08, 0x00},
	{0x3c09, 0x1c},
	{0x300c, 0x22}, /* 50/60div 300c[2:0] */
	{0x3c0a, 0x9c},
	{0x3c0b, 0x40},
};

struct ov5640_sensor ov5640_antibanding_50z_tbl[] = {
  /* Band 50Hz */
	{0x3c01, 0x80, INVMASK(0x80)},
	{0x3c00, 0x04},
};

struct ov5640_sensor ov5640_antibanding_60z_tbl[] = {
  /* Band 60Hz */
	{0x3c01, 0x80, INVMASK(0x80)},
	{0x3c00, 0x00},
};


/* Lens_shading */

struct ov5640_sensor ov5640_lens_shading_on_tbl[] = {
	/* @@ Lenc On(C) */
	{0x5000, 0x80, INVMASK(0x80)},
};

struct ov5640_sensor ov5640_lens_shading_off_tbl[] = {
	/*  Lenc Off */
	{0x5000, 0x00, INVMASK(0x80)},
};

/* Auto Focus Firmware-use 2011-08-24 firmware settings */
u8 ov5640_afinit_tbl[] = {
	0x80,	0x00,	0x02,	0x0b,	0x7b,	0x02,	0x07,	0xbd,	0xc2,
	0x01,	0x22,	0x22,	0x00,	0x02,	0x0b,	0x57,	0xe5,	0x1f,
	0x70,	0x72,	0xf5,	0x1e,	0xd2,	0x35,	0xff,	0xef,	0x25,
	0xe0,	0x24,	0x4b,	0xf8,	0xe4,	0xf6,	0x08,	0xf6,	0x0f,
	0xbf,	0x34,	0xf2,	0x90,	0x0e,	0x88,	0xe4,	0x93,	0xff,
	0xe5,	0x49,	0xc3,	0x9f,	0x50,	0x04,	0x7f,	0x05,	0x80,
	0x02,	0x7f,	0xfb,	0x78,	0xba,	0xa6,	0x07,	0x12,	0x0a,
	0xb4,	0x40,	0x04,	0x7f,	0x03,	0x80,	0x02,	0x7f,	0x30,
	0x78,	0xb9,	0xa6,	0x07,	0xe6,	0x18,	0xf6,	0x08,	0xe6,
	0x78,	0xb6,	0xf6,	0x78,	0xb9,	0xe6,	0x78,	0xb7,	0xf6,
	0x78,	0xbc,	0x76,	0x33,	0xe4,	0x08,	0xf6,	0x78,	0xb5,
	0x76,	0x01,	0x75,	0x48,	0x02,	0x78,	0xb3,	0xf6,	0x08,
	0xf6,	0x74,	0xff,	0x78,	0xbe,	0xf6,	0x08,	0xf6,	0x75,
	0x1f,	0x01,	0x78,	0xb9,	0xe6,	0x75,	0xf0,	0x05,	0xa4,
	0xf5,	0x49,	0x12,	0x08,	0x5b,	0xc2,	0x37,	0x22,	0x78,
	0xb5,	0xe6,	0xd3,	0x94,	0x00,	0x40,	0x02,	0x16,	0x22,
	0xe5,	0x1f,	0x64,	0x05,	0x70,	0x28,	0xf5,	0x1f,	0xc2,
	0x01,	0x78,	0xb6,	0xe6,	0x25,	0xe0,	0x24,	0x4b,	0xf8,
	0xe6,	0xfe,	0x08,	0xe6,	0xff,	0x78,	0x4b,	0xa6,	0x06,
	0x08,	0xa6,	0x07,	0xa2,	0x37,	0xe4,	0x33,	0xf5,	0x3c,
	0x90,	0x30,	0x28,	0xf0,	0x75,	0x1e,	0x10,	0xd2,	0x35,
	0x22,	0xe5,	0x49,	0x75,	0xf0,	0x05,	0x84,	0x78,	0xb9,
	0xf6,	0x90,	0x0e,	0x85,	0xe4,	0x93,	0xff,	0x25,	0xe0,
	0x24,	0x0a,	0xf8,	0xe6,	0xfc,	0x08,	0xe6,	0xfd,	0x78,
	0xb9,	0xe6,	0x25,	0xe0,	0x24,	0x4b,	0xf8,	0xa6,	0x04,
	0x08,	0xa6,	0x05,	0xef,	0x12,	0x0a,	0xbb,	0xd3,	0x78,
	0xb4,	0x96,	0xee,	0x18,	0x96,	0x40,	0x0d,	0x78,	0xb9,
	0xe6,	0x78,	0xb6,	0xf6,	0x78,	0xb3,	0xa6,	0x06,	0x08,
	0xa6,	0x07,	0x90,	0x0e,	0x85,	0xe4,	0x93,	0x12,	0x0a,
	0xbb,	0xc3,	0x78,	0xbf,	0x96,	0xee,	0x18,	0x96,	0x50,
	0x0d,	0x78,	0xb9,	0xe6,	0x78,	0xb7,	0xf6,	0x78,	0xbe,
	0xa6,	0x06,	0x08,	0xa6,	0x07,	0x78,	0xb3,	0xe6,	0xfe,
	0x08,	0xe6,	0xc3,	0x78,	0xbf,	0x96,	0xff,	0xee,	0x18,
	0x96,	0x78,	0xc0,	0xf6,	0x08,	0xa6,	0x07,	0x90,	0x0e,
	0x8a,	0xe4,	0x18,	0x12,	0x0a,	0x99,	0xc3,	0x33,	0xce,
	0x33,	0xce,	0xd8,	0xf9,	0xff,	0xd3,	0xed,	0x9f,	0xec,
	0x9e,	0x40,	0x02,	0xd2,	0x37,	0x78,	0xb9,	0xe6,	0x08,
	0x26,	0x08,	0xf6,	0xe5,	0x1f,	0x64,	0x01,	0x70,	0x55,
	0xe6,	0xc3,	0x78,	0xbd,	0x12,	0x0a,	0x8f,	0x40,	0x10,
	0x12,	0x0a,	0x8a,	0x50,	0x0b,	0x30,	0x37,	0x41,	0x78,
	0xb9,	0xe6,	0x78,	0xb6,	0x66,	0x60,	0x39,	0x12,	0x0a,
	0xb2,	0x40,	0x04,	0x7f,	0xfe,	0x80,	0x02,	0x7f,	0x02,
	0x78,	0xba,	0xa6,	0x07,	0x78,	0xb6,	0xe6,	0x24,	0x03,
	0x78,	0xbc,	0xf6,	0x78,	0xb6,	0xe6,	0x24,	0xfd,	0x78,
	0xbd,	0xf6,	0x12,	0x0a,	0xb2,	0x40,	0x06,	0x78,	0xbd,
	0xe6,	0xff,	0x80,	0x04,	0x78,	0xbc,	0xe6,	0xff,	0x78,
	0xbb,	0xa6,	0x07,	0x75,	0x1f,	0x02,	0x78,	0xb5,	0x76,
	0x01,	0x02,	0x02,	0x68,	0xe5,	0x1f,	0x64,	0x02,	0x60,
	0x03,	0x02,	0x02,	0x48,	0x78,	0xbb,	0xe6,	0xff,	0xc3,
	0x78,	0xbd,	0x12,	0x0a,	0x90,	0x40,	0x08,	0x12,	0x0a,
	0x8a,	0x50,	0x03,	0x02,	0x02,	0x46,	0x12,	0x0a,	0xb2,
	0x40,	0x04,	0x7f,	0xff,	0x80,	0x02,	0x7f,	0x01,	0x78,
	0xba,	0xa6,	0x07,	0x78,	0xb6,	0xe6,	0x04,	0x78,	0xbc,
	0xf6,	0x78,	0xb6,	0xe6,	0x14,	0x78,	0xbd,	0xf6,	0x18,
	0x12,	0x0a,	0xb4,	0x40,	0x04,	0xe6,	0xff,	0x80,	0x02,
	0x7f,	0x00,	0x78,	0xbc,	0xa6,	0x07,	0xd3,	0x08,	0xe6,
	0x64,	0x80,	0x94,	0x80,	0x40,	0x04,	0xe6,	0xff,	0x80,
	0x02,	0x7f,	0x00,	0x78,	0xbd,	0xa6,	0x07,	0xc3,	0x18,
	0xe6,	0x64,	0x80,	0x94,	0xb3,	0x50,	0x04,	0xe6,	0xff,
	0x80,	0x02,	0x7f,	0x33,	0x78,	0xbc,	0xa6,	0x07,	0xc3,
	0x08,	0xe6,	0x64,	0x80,	0x94,	0xb3,	0x50,	0x04,	0xe6,
	0xff,	0x80,	0x02,	0x7f,	0x33,	0x78,	0xbd,	0xa6,	0x07,
	0x12,	0x0a,	0xb2,	0x40,	0x06,	0x78,	0xbd,	0xe6,	0xff,
	0x80,	0x04,	0x78,	0xbc,	0xe6,	0xff,	0x78,	0xbb,	0xa6,
	0x07,	0x75,	0x1f,	0x03,	0x78,	0xb5,	0x76,	0x01,	0x80,
	0x20,	0xe5,	0x1f,	0x64,	0x03,	0x70,	0x26,	0x78,	0xbb,
	0xe6,	0xff,	0xc3,	0x78,	0xbd,	0x12,	0x0a,	0x90,	0x40,
	0x05,	0x12,	0x0a,	0x8a,	0x40,	0x09,	0x78,	0xb6,	0xe6,
	0x78,	0xbb,	0xf6,	0x75,	0x1f,	0x04,	0x78,	0xbb,	0xe6,
	0x75,	0xf0,	0x05,	0xa4,	0xf5,	0x49,	0x02,	0x08,	0x5b,
	0xe5,	0x1f,	0xb4,	0x04,	0x1d,	0x90,	0x0e,	0x89,	0xe4,
	0x78,	0xc0,	0x12,	0x0a,	0x99,	0xc3,	0x33,	0xce,	0x33,
	0xce,	0xd8,	0xf9,	0xff,	0xd3,	0xed,	0x9f,	0xec,	0x9e,
	0x40,	0x02,	0xd2,	0x37,	0x75,	0x1f,	0x05,	0x22,	0xef,
	0x8d,	0xf0,	0xa4,	0xa8,	0xf0,	0xcf,	0x8c,	0xf0,	0xa4,
	0x28,	0xce,	0x8d,	0xf0,	0xa4,	0x2e,	0xfe,	0x22,	0xbc,
	0x00,	0x0b,	0xbe,	0x00,	0x29,	0xef,	0x8d,	0xf0,	0x84,
	0xff,	0xad,	0xf0,	0x22,	0xe4,	0xcc,	0xf8,	0x75,	0xf0,
	0x08,	0xef,	0x2f,	0xff,	0xee,	0x33,	0xfe,	0xec,	0x33,
	0xfc,	0xee,	0x9d,	0xec,	0x98,	0x40,	0x05,	0xfc,	0xee,
	0x9d,	0xfe,	0x0f,	0xd5,	0xf0,	0xe9,	0xe4,	0xce,	0xfd,
	0x22,	0xed,	0xf8,	0xf5,	0xf0,	0xee,	0x84,	0x20,	0xd2,
	0x1c,	0xfe,	0xad,	0xf0,	0x75,	0xf0,	0x08,	0xef,	0x2f,
	0xff,	0xed,	0x33,	0xfd,	0x40,	0x07,	0x98,	0x50,	0x06,
	0xd5,	0xf0,	0xf2,	0x22,	0xc3,	0x98,	0xfd,	0x0f,	0xd5,
	0xf0,	0xea,	0x22,	0xe8,	0x8f,	0xf0,	0xa4,	0xcc,	0x8b,
	0xf0,	0xa4,	0x2c,	0xfc,	0xe9,	0x8e,	0xf0,	0xa4,	0x2c,
	0xfc,	0x8a,	0xf0,	0xed,	0xa4,	0x2c,	0xfc,	0xea,	0x8e,
	0xf0,	0xa4,	0xcd,	0xa8,	0xf0,	0x8b,	0xf0,	0xa4,	0x2d,
	0xcc,	0x38,	0x25,	0xf0,	0xfd,	0xe9,	0x8f,	0xf0,	0xa4,
	0x2c,	0xcd,	0x35,	0xf0,	0xfc,	0xeb,	0x8e,	0xf0,	0xa4,
	0xfe,	0xa9,	0xf0,	0xeb,	0x8f,	0xf0,	0xa4,	0xcf,	0xc5,
	0xf0,	0x2e,	0xcd,	0x39,	0xfe,	0xe4,	0x3c,	0xfc,	0xea,
	0xa4,	0x2d,	0xce,	0x35,	0xf0,	0xfd,	0xe4,	0x3c,	0xfc,
	0x22,	0x75,	0xf0,	0x08,	0x75,	0x82,	0x00,	0xef,	0x2f,
	0xff,	0xee,	0x33,	0xfe,	0xcd,	0x33,	0xcd,	0xcc,	0x33,
	0xcc,	0xc5,	0x82,	0x33,	0xc5,	0x82,	0x9b,	0xed,	0x9a,
	0xec,	0x99,	0xe5,	0x82,	0x98,	0x40,	0x0c,	0xf5,	0x82,
	0xee,	0x9b,	0xfe,	0xed,	0x9a,	0xfd,	0xec,	0x99,	0xfc,
	0x0f,	0xd5,	0xf0,	0xd6,	0xe4,	0xce,	0xfb,	0xe4,	0xcd,
	0xfa,	0xe4,	0xcc,	0xf9,	0xa8,	0x82,	0x22,	0xb8,	0x00,
	0xc1,	0xb9,	0x00,	0x59,	0xba,	0x00,	0x2d,	0xec,	0x8b,
	0xf0,	0x84,	0xcf,	0xce,	0xcd,	0xfc,	0xe5,	0xf0,	0xcb,
	0xf9,	0x78,	0x18,	0xef,	0x2f,	0xff,	0xee,	0x33,	0xfe,
	0xed,	0x33,	0xfd,	0xec,	0x33,	0xfc,	0xeb,	0x33,	0xfb,
	0x10,	0xd7,	0x03,	0x99,	0x40,	0x04,	0xeb,	0x99,	0xfb,
	0x0f,	0xd8,	0xe5,	0xe4,	0xf9,	0xfa,	0x22,	0x78,	0x18,
	0xef,	0x2f,	0xff,	0xee,	0x33,	0xfe,	0xed,	0x33,	0xfd,
	0xec,	0x33,	0xfc,	0xc9,	0x33,	0xc9,	0x10,	0xd7,	0x05,
	0x9b,	0xe9,	0x9a,	0x40,	0x07,	0xec,	0x9b,	0xfc,	0xe9,
	0x9a,	0xf9,	0x0f,	0xd8,	0xe0,	0xe4,	0xc9,	0xfa,	0xe4,
	0xcc,	0xfb,	0x22,	0x75,	0xf0,	0x10,	0xef,	0x2f,	0xff,
	0xee,	0x33,	0xfe,	0xed,	0x33,	0xfd,	0xcc,	0x33,	0xcc,
	0xc8,	0x33,	0xc8,	0x10,	0xd7,	0x07,	0x9b,	0xec,	0x9a,
	0xe8,	0x99,	0x40,	0x0a,	0xed,	0x9b,	0xfd,	0xec,	0x9a,
	0xfc,	0xe8,	0x99,	0xf8,	0x0f,	0xd5,	0xf0,	0xda,	0xe4,
	0xcd,	0xfb,	0xe4,	0xcc,	0xfa,	0xe4,	0xc8,	0xf9,	0x22,
	0xeb,	0x9f,	0xf5,	0xf0,	0xea,	0x9e,	0x42,	0xf0,	0xe9,
	0x9d,	0x42,	0xf0,	0xe8,	0x9c,	0x45,	0xf0,	0x22,	0xe8,
	0x60,	0x0f,	0xef,	0xc3,	0x33,	0xff,	0xee,	0x33,	0xfe,
	0xed,	0x33,	0xfd,	0xec,	0x33,	0xfc,	0xd8,	0xf1,	0x22,
	0xe4,	0x93,	0xfc,	0x74,	0x01,	0x93,	0xfd,	0x74,	0x02,
	0x93,	0xfe,	0x74,	0x03,	0x93,	0xff,	0x22,	0xe6,	0xfb,
	0x08,	0xe6,	0xf9,	0x08,	0xe6,	0xfa,	0x08,	0xe6,	0xcb,
	0xf8,	0x22,	0xec,	0xf6,	0x08,	0xed,	0xf6,	0x08,	0xee,
	0xf6,	0x08,	0xef,	0xf6,	0x22,	0xa4,	0x25,	0x82,	0xf5,
	0x82,	0xe5,	0xf0,	0x35,	0x83,	0xf5,	0x83,	0x22,	0xd0,
	0x83,	0xd0,	0x82,	0xf8,	0xe4,	0x93,	0x70,	0x12,	0x74,
	0x01,	0x93,	0x70,	0x0d,	0xa3,	0xa3,	0x93,	0xf8,	0x74,
	0x01,	0x93,	0xf5,	0x82,	0x88,	0x83,	0xe4,	0x73,	0x74,
	0x02,	0x93,	0x68,	0x60,	0xef,	0xa3,	0xa3,	0xa3,	0x80,
	0xdf,	0x90,	0x38,	0x04,	0x78,	0x4f,	0x12,	0x09,	0x50,
	0x90,	0x38,	0x00,	0xe0,	0xfe,	0xa3,	0xe0,	0xfd,	0xed,
	0xff,	0xc3,	0x12,	0x09,	0x09,	0x90,	0x38,	0x10,	0x12,
	0x08,	0xfd,	0x90,	0x38,	0x06,	0x78,	0x51,	0x12,	0x09,
	0x50,	0x90,	0x38,	0x02,	0xe0,	0xfe,	0xa3,	0xe0,	0xfd,
	0xed,	0xff,	0xc3,	0x12,	0x09,	0x09,	0x90,	0x38,	0x12,
	0x12,	0x08,	0xfd,	0xa3,	0xe0,	0xb4,	0x31,	0x07,	0x78,
	0x4f,	0x79,	0x4f,	0x12,	0x09,	0x66,	0x90,	0x38,	0x14,
	0xe0,	0xb4,	0x71,	0x15,	0x78,	0x4f,	0xe6,	0xfe,	0x08,
	0xe6,	0x78,	0x02,	0xce,	0xc3,	0x13,	0xce,	0x13,	0xd8,
	0xf9,	0x79,	0x50,	0xf7,	0xee,	0x19,	0xf7,	0x90,	0x38,
	0x15,	0xe0,	0xb4,	0x31,	0x07,	0x78,	0x51,	0x79,	0x51,
	0x12,	0x09,	0x66,	0x90,	0x38,	0x15,	0xe0,	0xb4,	0x71,
	0x15,	0x78,	0x51,	0xe6,	0xfe,	0x08,	0xe6,	0x78,	0x02,
	0xce,	0xc3,	0x13,	0xce,	0x13,	0xd8,	0xf9,	0x79,	0x52,
	0xf7,	0xee,	0x19,	0xf7,	0x79,	0x4f,	0x12,	0x09,	0x38,
	0x09,	0x12,	0x09,	0x38,	0xaf,	0x45,	0x12,	0x08,	0xee,
	0x7d,	0x50,	0x12,	0x02,	0xa9,	0x78,	0x57,	0xa6,	0x06,
	0x08,	0xa6,	0x07,	0xaf,	0x43,	0x12,	0x08,	0xee,	0x7d,
	0x50,	0x12,	0x02,	0xa9,	0x78,	0x53,	0xa6,	0x06,	0x08,
	0xa6,	0x07,	0xaf,	0x46,	0x78,	0x51,	0x12,	0x08,	0xf0,
	0x7d,	0x3c,	0x12,	0x02,	0xa9,	0x78,	0x59,	0xa6,	0x06,
	0x08,	0xa6,	0x07,	0xaf,	0x44,	0x7e,	0x00,	0x78,	0x51,
	0x12,	0x08,	0xf2,	0x7d,	0x3c,	0x12,	0x02,	0xa9,	0x78,
	0x55,	0xa6,	0x06,	0x08,	0xa6,	0x07,	0xc3,	0x78,	0x58,
	0xe6,	0x94,	0x08,	0x18,	0xe6,	0x94,	0x00,	0x50,	0x05,
	0x76,	0x00,	0x08,	0x76,	0x08,	0xc3,	0x78,	0x5a,	0xe6,
	0x94,	0x08,	0x18,	0xe6,	0x94,	0x00,	0x50,	0x05,	0x76,
	0x00,	0x08,	0x76,	0x08,	0x78,	0x57,	0x12,	0x09,	0x25,
	0xff,	0xd3,	0x78,	0x54,	0xe6,	0x9f,	0x18,	0xe6,	0x9e,
	0x40,	0x0e,	0x78,	0x57,	0xe6,	0x13,	0xfe,	0x08,	0xe6,
	0x78,	0x54,	0x12,	0x09,	0x5b,	0x80,	0x04,	0x7e,	0x00,
	0x7f,	0x00,	0x78,	0x5b,	0x12,	0x09,	0x1d,	0xff,	0xd3,
	0x78,	0x56,	0xe6,	0x9f,	0x18,	0xe6,	0x9e,	0x40,	0x0e,
	0x78,	0x59,	0xe6,	0x13,	0xfe,	0x08,	0xe6,	0x78,	0x56,
	0x12,	0x09,	0x5b,	0x80,	0x04,	0x7e,	0x00,	0x7f,	0x00,
	0xe4,	0xfc,	0xfd,	0x78,	0x5f,	0x12,	0x04,	0x5c,	0x78,
	0x57,	0x12,	0x09,	0x25,	0x78,	0x54,	0x26,	0xff,	0xee,
	0x18,	0x36,	0xfe,	0x78,	0x63,	0x12,	0x09,	0x1d,	0x78,
	0x56,	0x26,	0xff,	0xee,	0x18,	0x36,	0xfe,	0xe4,	0xfc,
	0xfd,	0x78,	0x67,	0x12,	0x04,	0x5c,	0x12,	0x09,	0x2d,
	0x78,	0x63,	0x12,	0x04,	0x4f,	0xd3,	0x12,	0x04,	0x1b,
	0x40,	0x08,	0x12,	0x09,	0x2d,	0x78,	0x63,	0x12,	0x04,
	0x5c,	0x78,	0x51,	0x12,	0x09,	0x2f,	0x78,	0x67,	0x12,
	0x04,	0x4f,	0xd3,	0x12,	0x04,	0x1b,	0x40,	0x0a,	0x78,
	0x51,	0x12,	0x09,	0x2f,	0x78,	0x67,	0x12,	0x04,	0x5c,
	0xe4,	0xfd,	0x78,	0x5e,	0x12,	0x09,	0x48,	0x24,	0x01,
	0x12,	0x09,	0x11,	0x78,	0x62,	0x12,	0x09,	0x48,	0x24,
	0x02,	0x12,	0x09,	0x11,	0x78,	0x66,	0x12,	0x09,	0x48,
	0x24,	0x03,	0x12,	0x09,	0x11,	0x78,	0x6a,	0x12,	0x09,
	0x48,	0x24,	0x04,	0x12,	0x09,	0x11,	0x0d,	0xbd,	0x05,
	0xd4,	0xc2,	0x0e,	0xc2,	0x06,	0x22,	0x85,	0x08,	0x41,
	0x90,	0x30,	0x24,	0xe0,	0xf5,	0x3d,	0xa3,	0xe0,	0xf5,
	0x3e,	0xa3,	0xe0,	0xf5,	0x3f,	0xa3,	0xe0,	0xf5,	0x40,
	0xa3,	0xe0,	0xf5,	0x3c,	0xd2,	0x34,	0xe5,	0x41,	0x12,
	0x04,	0x74,	0x06,	0xc7,	0x03,	0x06,	0xcb,	0x04,	0x06,
	0xd1,	0x07,	0x06,	0xda,	0x08,	0x06,	0xeb,	0x12,	0x07,
	0x03,	0x18,	0x07,	0x19,	0x19,	0x06,	0xee,	0x1a,	0x06,
	0xfa,	0x1b,	0x07,	0x3e,	0x80,	0x07,	0x43,	0x81,	0x07,
	0xa1,	0x8f,	0x07,	0x90,	0x90,	0x07,	0xa1,	0x91,	0x07,
	0xa1,	0x92,	0x07,	0xa1,	0x93,	0x07,	0xa1,	0x94,	0x07,
	0xa1,	0x98,	0x07,	0x9e,	0x9f,	0x00,	0x00,	0x07,	0xbc,
	0x12,	0x0a,	0xf4,	0x22,	0x12,	0x0a,	0xf4,	0xd2,	0x03,
	0x22,	0xa2,	0x37,	0xe4,	0x33,	0xf5,	0x3c,	0x02,	0x07,
	0xa1,	0xc2,	0x01,	0xc2,	0x02,	0xc2,	0x03,	0x12,	0x09,
	0x70,	0x75,	0x1e,	0x70,	0xd2,	0x35,	0x02,	0x07,	0xa1,
	0x02,	0x07,	0x8b,	0x85,	0x40,	0x48,	0x85,	0x3c,	0x49,
	0x12,	0x08,	0x5b,	0x02,	0x07,	0xa1,	0x85,	0x48,	0x40,
	0x85,	0x49,	0x3c,	0x02,	0x07,	0xa1,	0xe4,	0xf5,	0x22,
	0xf5,	0x23,	0x85,	0x40,	0x31,	0x85,	0x3f,	0x30,	0x85,
	0x3e,	0x2f,	0x85,	0x3d,	0x2e,	0x12,	0x0a,	0xc6,	0x80,
	0x1f,	0x75,	0x22,	0x00,	0x75,	0x23,	0x01,	0x74,	0xff,
	0xf5,	0x2d,	0xf5,	0x2c,	0xf5,	0x2b,	0xf5,	0x2a,	0x12,
	0x0a,	0xc6,	0x85,	0x2d,	0x40,	0x85,	0x2c,	0x3f,	0x85,
	0x2b,	0x3e,	0x85,	0x2a,	0x3d,	0xe4,	0xf5,	0x3c,	0x02,
	0x07,	0xa1,	0x12,	0x0b,	0x3d,	0x80,	0x5e,	0x85,	0x3d,
	0x43,	0x85,	0x3e,	0x44,	0xe5,	0x45,	0xc3,	0x13,	0xff,
	0xe5,	0x43,	0xc3,	0x9f,	0x50,	0x02,	0x8f,	0x43,	0xe5,
	0x46,	0xc3,	0x13,	0xff,	0xe5,	0x44,	0xc3,	0x9f,	0x50,
	0x02,	0x8f,	0x44,	0xe5,	0x45,	0xc3,	0x13,	0xff,	0xfd,
	0xe5,	0x43,	0x90,	0x0e,	0x7f,	0x12,	0x0b,	0x10,	0x40,
	0x04,	0xee,	0x9f,	0xf5,	0x43,	0xe5,	0x46,	0xc3,	0x13,
	0xff,	0xfd,	0xe5,	0x44,	0x90,	0x0e,	0x80,	0x12,	0x0b,
	0x10,	0x40,	0x04,	0xee,	0x9f,	0xf5,	0x44,	0x12,	0x04,
	0x9a,	0x80,	0x11,	0x85,	0x40,	0x46,	0x85,	0x3f,	0x45,
	0x85,	0x3e,	0x44,	0x85,	0x3d,	0x43,	0x80,	0x03,	0x02,
	0x04,	0x9a,	0x90,	0x30,	0x24,	0xe5,	0x3d,	0xf0,	0xa3,
	0xe5,	0x3e,	0xf0,	0xa3,	0xe5,	0x3f,	0xf0,	0xa3,	0xe5,
	0x40,	0xf0,	0xa3,	0xe5,	0x3c,	0xf0,	0x90,	0x30,	0x23,
	0xe4,	0xf0,	0x22,	0xc0,	0xe0,	0xc0,	0x83,	0xc0,	0x82,
	0xc0,	0xd0,	0x90,	0x3f,	0x0c,	0xe0,	0xf5,	0x32,	0xe5,
	0x32,	0x30,	0xe3,	0x4c,	0x30,	0x36,	0x3e,	0x90,	0x60,
	0x19,
	0xe0,
	0xf5,
	0x0a,
	0xa3,
	0xe0,
	0xf5,
	0x0b,
	0x90,
	0x60,
	0x1d,
	0xe0,
	0xf5,
	0x14,
	0xa3,
	0xe0,
	0xf5,
	0x15,
	0x30,
	0x01,
	0x06,
	0x30,
	0x33,
	0x03,
	0xd3,
	0x80,
	0x01,
	0xc3,
	0x92,
	0x09,
	0x30,
	0x02,
	0x06,
	0x30,
	0x33,
	0x03,
	0xd3,
	0x80,
	0x01,
	0xc3,
	0x92,
	0x0a,
	0x30,
	0x33,
	0x0c,
	0x30,
	0x03,
	0x09,
	0x20,
	0x02,
	0x06,
	0x20,
	0x01,
	0x03,
	0xd3,
	0x80,
	0x01,
	0xc3,
	0x92,
	0x0b,
	0x90,
	0x30,
	0x01,
	0xe0,
	0x44,
	0x40,
	0xf0,
	0xe0,
	0x54,
	0xbf,
	0xf0,
	0xe5,
	0x32,
	0x30,
	0xe1,
	0x14,
	0x30,
	0x34,
	0x11,
	0x90,
	0x30,
	0x22,
	0xe0,
	0xf5,
	0x08,
	0xe4,
	0xf0,
	0x30,
	0x00,
	0x03,
	0xd3,
	0x80,
	0x01,
	0xc3,
	0x92,
	0x08,
	0xe5,
	0x32,
	0x30,
	0xe5,
	0x12,
	0x90,
	0x56,
	0xa1,
	0xe0,
	0xf5,
	0x09,
	0x30,
	0x31,
	0x09,
	0x30,
	0x05,
	0x03,
	0xd3,
	0x80,
	0x01,
	0xc3,
	0x92,
	0x0d,
	0x90,
	0x3f,
	0x0c,
	0xe5,
	0x32,
	0xf0,
	0xd0,
	0xd0,
	0xd0,
	0x82,
	0xd0,
	0x83,
	0xd0,
	0xe0,
	0x32,
	0x90,
	0x0e,
	0x7d,
	0xe4,
	0x93,
	0xfe,
	0x74,
	0x01,
	0x93,
	0xff,
	0xc3,
	0x90,
	0x0e,
	0x7b,
	0x74,
	0x01,
	0x93,
	0x9f,
	0xff,
	0xe4,
	0x93,
	0x9e,
	0xfe,
	0xe4,
	0x8f,
	0x3b,
	0x8e,
	0x3a,
	0xf5,
	0x39,
	0xf5,
	0x38,
	0xab,
	0x3b,
	0xaa,
	0x3a,
	0xa9,
	0x39,
	0xa8,
	0x38,
	0xaf,
	0x49,
	0xfc,
	0xfd,
	0xfe,
	0x12,
	0x02,
	0xfe,
	0x12,
	0x0b,
	0x22,
	0xe4,
	0x7b,
	0xff,
	0xfa,
	0xf9,
	0xf8,
	0x12,
	0x03,
	0x89,
	0x12,
	0x0b,
	0x22,
	0x90,
	0x0e,
	0x69,
	0xe4,
	0x12,
	0x0b,
	0x37,
	0x12,
	0x0b,
	0x22,
	0xe4,
	0x85,
	0x48,
	0x37,
	0xf5,
	0x36,
	0xf5,
	0x35,
	0xf5,
	0x34,
	0xaf,
	0x37,
	0xae,
	0x36,
	0xad,
	0x35,
	0xac,
	0x34,
	0xa3,
	0x12,
	0x0b,
	0x37,
	0x8f,
	0x37,
	0x8e,
	0x36,
	0x8d,
	0x35,
	0x8c,
	0x34,
	0xe5,
	0x3b,
	0x45,
	0x37,
	0xf5,
	0x3b,
	0xe5,
	0x3a,
	0x45,
	0x36,
	0xf5,
	0x3a,
	0xe5,
	0x39,
	0x45,
	0x35,
	0xf5,
	0x39,
	0xe5,
	0x38,
	0x45,
	0x34,
	0xf5,
	0x38,
	0xe4,
	0xf5,
	0x22,
	0xf5,
	0x23,
	0x85,
	0x3b,
	0x31,
	0x85,
	0x3a,
	0x30,
	0x85,
	0x39,
	0x2f,
	0x85,
	0x38,
	0x2e,
	0x02,
	0x0a,
	0xc6,
	0x78,
	0x4f,
	0x7e,
	0x00,
	0xe6,
	0xfc,
	0x08,
	0xe6,
	0xfd,
	0x12,
	0x02,
	0x97,
	0x7c,
	0x00,
	0x22,
	0xe0,
	0xa3,
	0xe0,
	0x75,
	0xf0,
	0x02,
	0xa4,
	0xff,
	0xae,
	0xf0,
	0xc3,
	0x08,
	0xe6,
	0x9f,
	0xf6,
	0x18,
	0xe6,
	0x9e,
	0xf6,
	0x22,
	0xff,
	0xe5,
	0xf0,
	0x34,
	0x60,
	0x8f,
	0x82,
	0xf5,
	0x83,
	0xec,
	0xf0,
	0x22,
	0xe4,
	0xfc,
	0xfd,
	0x12,
	0x04,
	0x5c,
	0x78,
	0x59,
	0xe6,
	0xc3,
	0x13,
	0xfe,
	0x08,
	0xe6,
	0x13,
	0x22,
	0x78,
	0x4f,
	0xe6,
	0xfe,
	0x08,
	0xe6,
	0xff,
	0xe4,
	0xfc,
	0xfd,
	0x22,
	0xe7,
	0xc4,
	0xf8,
	0x54,
	0xf0,
	0xc8,
	0x68,
	0xf7,
	0x09,
	0xe7,
	0xc4,
	0x54,
	0x0f,
	0x48,
	0xf7,
	0x22,
	0xe6,
	0xfc,
	0xed,
	0x75,
	0xf0,
	0x04,
	0xa4,
	0x22,
	0xe0,
	0xfe,
	0xa3,
	0xe0,
	0xfd,
	0xee,
	0xf6,
	0xed,
	0x08,
	0xf6,
	0x22,
	0x13,
	0xff,
	0xc3,
	0xe6,
	0x9f,
	0xff,
	0x18,
	0xe6,
	0x9e,
	0xfe,
	0x22,
	0xe6,
	0xc3,
	0x13,
	0xf7,
	0x08,
	0xe6,
	0x13,
	0x09,
	0xf7,
	0x22,
	0xe4,
	0xf5,
	0x49,
	0x90,
	0x0e,
	0x77,
	0x93,
	0xff,
	0xe4,
	0x8f,
	0x37,
	0xf5,
	0x36,
	0xf5,
	0x35,
	0xf5,
	0x34,
	0xaf,
	0x37,
	0xae,
	0x36,
	0xad,
	0x35,
	0xac,
	0x34,
	0x90,
	0x0e,
	0x6a,
	0x12,
	0x0b,
	0x37,
	0x8f,
	0x37,
	0x8e,
	0x36,
	0x8d,
	0x35,
	0x8c,
	0x34,
	0x90,
	0x0e,
	0x72,
	0x12,
	0x04,
	0x3f,
	0xef,
	0x45,
	0x37,
	0xf5,
	0x37,
	0xee,
	0x45,
	0x36,
	0xf5,
	0x36,
	0xed,
	0x45,
	0x35,
	0xf5,
	0x35,
	0xec,
	0x45,
	0x34,
	0xf5,
	0x34,
	0xe4,
	0xf5,
	0x22,
	0xf5,
	0x23,
	0x85,
	0x37,
	0x31,
	0x85,
	0x36,
	0x30,
	0x85,
	0x35,
	0x2f,
	0x85,
	0x34,
	0x2e,
	0x12,
	0x0a,
	0xc6,
	0xe4,
	0xf5,
	0x22,
	0xf5,
	0x23,
	0x90,
	0x0e,
	0x72,
	0x12,
	0x0b,
	0x2b,
	0x12,
	0x0a,
	0xc6,
	0xe4,
	0xf5,
	0x22,
	0xf5,
	0x23,
	0x90,
	0x0e,
	0x6e,
	0x12,
	0x0b,
	0x2b,
	0x02,
	0x0a,
	0xc6,
	0x75,
	0x89,
	0x03,
	0x75,
	0xa8,
	0x01,
	0x75,
	0xb8,
	0x04,
	0x75,
	0x34,
	0xff,
	0x75,
	0x35,
	0x0e,
	0x75,
	0x36,
	0x15,
	0x75,
	0x37,
	0x0d,
	0x12,
	0x0a,
	0x4a,
	0x12,
	0x00,
	0x09,
	0x12,
	0x0b,
	0x3d,
	0x12,
	0x00,
	0x06,
	0xd2,
	0x00,
	0xd2,
	0x34,
	0xd2,
	0xaf,
	0x75,
	0x34,
	0xff,
	0x75,
	0x35,
	0x0e,
	0x75,
	0x36,
	0x49,
	0x75,
	0x37,
	0x03,
	0x12,
	0x0a,
	0x4a,
	0x30,
	0x08,
	0x09,
	0xc2,
	0x34,
	0x12,
	0x06,
	0x6a,
	0xc2,
	0x08,
	0xd2,
	0x34,
	0x30,
	0x09,
	0x09,
	0xc2,
	0x36,
	0x12,
	0x00,
	0x0e,
	0xc2,
	0x09,
	0xd2,
	0x36,
	0x30,
	0x0e,
	0x03,
	0x12,
	0x04,
	0x9a,
	0x30,
	0x35,
	0xdf,
	0x90,
	0x30,
	0x29,
	0xe5,
	0x1e,
	0xf0,
	0xb4,
	0x10,
	0x05,
	0x90,
	0x30,
	0x23,
	0xe4,
	0xf0,
	0xc2,
	0x35,
	0x80,
	0xcd,
	0xae,
	0x35,
	0xaf,
	0x36,
	0xe4,
	0xfd,
	0xed,
	0xc3,
	0x95,
	0x37,
	0x50,
	0x33,
	0x12,
	0x0b,
	0x87,
	0xe4,
	0x93,
	0xf5,
	0x38,
	0x74,
	0x01,
	0x93,
	0xf5,
	0x39,
	0x45,
	0x38,
	0x60,
	0x23,
	0x85,
	0x39,
	0x82,
	0x85,
	0x38,
	0x83,
	0xe0,
	0xfc,
	0x12,
	0x0b,
	0x87,
	0x74,
	0x03,
	0x93,
	0x52,
	0x04,
	0x12,
	0x0b,
	0x87,
	0x74,
	0x02,
	0x93,
	0x42,
	0x04,
	0x85,
	0x39,
	0x82,
	0x85,
	0x38,
	0x83,
	0xec,
	0xf0,
	0x0d,
	0x80,
	0xc7,
	0x22,
	0x78,
	0xbb,
	0xe6,
	0xd3,
	0x08,
	0xff,
	0xe6,
	0x64,
	0x80,
	0xf8,
	0xef,
	0x64,
	0x80,
	0x98,
	0x22,
	0x93,
	0xff,
	0x7e,
	0x00,
	0xe6,
	0xfc,
	0x08,
	0xe6,
	0xfd,
	0x12,
	0x02,
	0x97,
	0xac,
	0x06,
	0xad,
	0x07,
	0x78,
	0xb3,
	0xe6,
	0xfe,
	0x08,
	0xe6,
	0x78,
	0x03,
	0x22,
	0x78,
	0xba,
	0xd3,
	0xe6,
	0x64,
	0x80,
	0x94,
	0x80,
	0x22,
	0x25,
	0xe0,
	0x24,
	0x0a,
	0xf8,
	0xe6,
	0xfe,
	0x08,
	0xe6,
	0xff,
	0x22,
	0xa2,
	0xaf,
	0x92,
	0x32,
	0xc2,
	0xaf,
	0xe5,
	0x23,
	0x45,
	0x22,
	0x90,
	0x0e,
	0x5d,
	0x60,
	0x0e,
	0x12,
	0x0b,
	0x70,
	0xe0,
	0xf5,
	0x2c,
	0x12,
	0x0b,
	0x6d,
	0xe0,
	0xf5,
	0x2d,
	0x80,
	0x0c,
	0x12,
	0x0b,
	0x70,
	0xe5,
	0x30,
	0xf0,
	0x12,
	0x0b,
	0x6d,
	0xe5,
	0x31,
	0xf0,
	0xa2,
	0x32,
	0x92,
	0xaf,
	0x22,
	0xd2,
	0x01,
	0xc2,
	0x02,
	0xe4,
	0xf5,
	0x1f,
	0xf5,
	0x1e,
	0xd2,
	0x35,
	0xd2,
	0x33,
	0xd2,
	0x36,
	0xd2,
	0x01,
	0xc2,
	0x02,
	0xf5,
	0x1f,
	0xf5,
	0x1e,
	0xd2,
	0x35,
	0xd2,
	0x33,
	0x22,
	0x2d,
	0xfd,
	0xe4,
	0x33,
	0xfc,
	0xe4,
	0x93,
	0xfe,
	0xfb,
	0xd3,
	0xed,
	0x9b,
	0x74,
	0x80,
	0xf8,
	0x6c,
	0x98,
	0x22,
	0x8f,
	0x3b,
	0x8e,
	0x3a,
	0x8d,
	0x39,
	0x8c,
	0x38,
	0x22,
	0x12,
	0x04,
	0x3f,
	0x8f,
	0x31,
	0x8e,
	0x30,
	0x8d,
	0x2f,
	0x8c,
	0x2e,
	0x22,
	0x93,
	0xf9,
	0xf8,
	0x02,
	0x04,
	0x2c,
	0x90,
	0x0e,
	0x81,
	0x12,
	0x04,
	0x3f,
	0x8f,
	0x46,
	0x8e,
	0x45,
	0x8d,
	0x44,
	0x8c,
	0x43,
	0xd2,
	0x06,
	0x30,
	0x06,
	0x03,
	0xd3,
	0x80,
	0x01,
	0xc3,
	0x92,
	0x0e,
	0x22,
	0xc0,
	0xe0,
	0xc0,
	0x83,
	0xc0,
	0x82,
	0x90,
	0x3f,
	0x0d,
	0xe0,
	0xf5,
	0x33,
	0xe5,
	0x33,
	0xf0,
	0xd0,
	0x82,
	0xd0,
	0x83,
	0xd0,
	0xe0,
	0x32,
	0x90,
	0x0e,
	0x5f,
	0xe4,
	0x93,
	0xfe,
	0x74,
	0x01,
	0x93,
	0xf5,
	0x82,
	0x8e,
	0x83,
	0x22,
	0x78,
	0x7f,
	0xe4,
	0xf6,
	0xd8,
	0xfd,
	0x75,
	0x81,
	0xca,
	0x02,
	0x09,
	0xe1,
	0x8f,
	0x82,
	0x8e,
	0x83,
	0x75,
	0xf0,
	0x04,
	0xed,
	0x02,
	0x04,
	0x68,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x11,
	0x07,
	0x21,
	0x15,
	0x29,
	0x13,
	0x4f,
	0x56,
	0x54,
	0x20,
	0x20,
	0x20,
	0x20,
	0x20,
	0x20,
	0x01,
	0x10,
	0x00,
	0x56,
	0x40,
	0x1a,
	0x30,
	0x29,
	0x7e,
	0x00,
	0x30,
	0x04,
	0x20,
	0xdf,
	0x30,
	0x05,
	0x40,
	0xbf,
	0x50,
	0x03,
	0x00,
	0xfd,
	0x50,
	0x27,
	0x01,
	0xfe,
	0x60,
	0x00,
	0x11,
	0x00,
	0x3f,
	0x05,
	0x30,
	0x00,
	0x3f,
	0x06,
	0x22,
	0x00,
	0x3f,
	0x01,
	0x2a,
	0x00,
	0x3f,
	0x02,
	0x00,
	0x00,
	0x36,
	0x06,
	0x07,
	0x00,
	0x3f,
	0x0b,
	0x0f,
	0xf0,
	0x00,
	0x00,
	0x00,
	0x00,
	0x30,
	0x01,
	0x40,
	0xbf,
	0x30,
	0x01,
	0x00,
	0xbf,
	0x30,
	0x29,
	0x70,
	0x00,
	0x3a,
	0x00,
	0x00,
	0xff,
	0x3a,
	0x00,
	0x00,
	0xff,
	0x36,
	0x03,
	0x36,
	0x02,
	0x41,
	0x44,
	0x58,
	0x20,
	0x18,
	0x10,
	0x0a,
	0x04,
	0x04,
	0x00,
	0x03,
	0xff,
	0x64,
	0x00,
	0x00,
	0x80,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x02,
	0x04,
	0x06,
	0x00,
	0x03,
	0x98,
	0x00,
	0xcc,
	0x50,
	0x3c,
	0x28,
	0x1e,
	0x10,
	0x10,
	0x00,
	0x00,
	0x00,
	0x6e,
	0x30,
	0x28,
	0x00,
	0xa5,
	0x5a,
	0x00,
};

#endif /* CAMSENSOR_OV5640 */
