#ifndef _LINUX_DCCP_H
#define _LINUX_DCCP_H

#include <linux/types.h>
#include <asm/byteorder.h>

/* Structure describing an Internet (DCCP) socket address. */
struct sockaddr_dccp {
	__u16	sdccp_family;	/* Address family   */
	__u16	sdccp_port;	/* Port number	    */
	__u32	sdccp_addr;	/* Internet address */
	__u32	sdccp_service;	/* Service	    */
	/* Pad to size of `struct sockaddr': 16 bytes . */
	__u32	sdccp_pad;
};

/**
 * struct dccp_hdr - generic part of DCCP packet header
 *
 * @dccph_sport - Relevant port on the endpoint that sent this packet
 * @dccph_dport - Relevant port on the other endpoint
 * @dccph_doff - Data Offset from the start of the DCCP header, in 32-bit words
 * @dccph_ccval - Used by the HC-Sender CCID
 * @dccph_cscov - Parts of the packet that are covered by the Checksum field
 * @dccph_checksum - Internet checksum, depends on dccph_cscov
 * @dccph_x - 0 = 24 bit sequence number, 1 = 48
 * @dccph_type - packet type, see DCCP_PKT_ prefixed macros
 * @dccph_seq - sequence number high or low order 24 bits, depends on dccph_x
 */
struct dccp_hdr {
	__u16	dccph_sport,
		dccph_dport;
	__u8	dccph_doff;
#if defined(__LITTLE_ENDIAN_BITFIELD)
	__u8	dccph_cscov:4,
		dccph_ccval:4;
#elif defined(__BIG_ENDIAN_BITFIELD)
	__u8	dccph_ccval:4,
		dccph_cscov:4;
#else
#error  "Adjust your <asm/byteorder.h> defines"
#endif
	__u16	dccph_checksum;
#if defined(__LITTLE_ENDIAN_BITFIELD)
	__u32	dccph_x:1,
		dccph_type:4,
		dccph_reserved:3,
		dccph_seq:24;
#elif defined(__BIG_ENDIAN_BITFIELD)
	__u32	dccph_reserved:3,
		dccph_type:4,
		dccph_x:1,
		dccph_seq:24;
#else
#error  "Adjust your <asm/byteorder.h> defines"
#endif
};

/**
 * struct dccp_hdr_ext - the low bits of a 48 bit seq packet
 *
 * @dccph_seq_low - low 24 bits of a 48 bit seq packet
 */
struct dccp_hdr_ext {
	__u32	dccph_seq_low;
};

/**
 * struct dccp_hdr_request - Conection initiation request header
 *
 * @dccph_req_service - Service to which the client app wants to connect
 * @dccph_req_options - list of options (must be a multiple of 32 bits
 */
struct dccp_hdr_request {
	__u32	dccph_req_service;
};
/**
 * struct dccp_hdr_ack_bits - acknowledgment bits common to most packets
 *
 * @dccph_resp_ack_nr_high - 48 bit ack number high order bits, contains GSR
 * @dccph_resp_ack_nr_low - 48 bit ack number low order bits, contains GSR
 */
struct dccp_hdr_ack_bits {
	__u32	dccph_reserved1:8,
		dccph_ack_nr_high:24;
	__u32	dccph_ack_nr_low;
};
/**
 * struct dccp_hdr_response - Conection initiation response header
 *
 * @dccph_resp_ack_nr_high - 48 bit ack number high order bits, contains GSR
 * @dccph_resp_ack_nr_low - 48 bit ack number low order bits, contains GSR
 * @dccph_resp_service - Echoes the Service Code on a received DCCP-Request
 * @dccph_resp_options - list of options (must be a multiple of 32 bits
 */
struct dccp_hdr_response {
	struct dccp_hdr_ack_bits	dccph_resp_ack;
	__u32				dccph_resp_service;
};

/**
 * struct dccp_hdr_reset - Unconditionally shut down a connection
 *
 * @dccph_reset_service - Echoes the Service Code on a received DCCP-Request
 * @dccph_reset_options - list of options (must be a multiple of 32 bits
 */
struct dccp_hdr_reset {
	struct dccp_hdr_ack_bits	dccph_reset_ack;
	__u8				dccph_reset_code,
					dccph_reset_data[3];
};

enum dccp_pkt_type {
	DCCP_PKT_REQUEST = 0,
	DCCP_PKT_RESPONSE,
	DCCP_PKT_DATA,
	DCCP_PKT_ACK,
	DCCP_PKT_DATAACK,
	DCCP_PKT_CLOSEREQ,
	DCCP_PKT_CLOSE,
	DCCP_PKT_RESET,
	DCCP_PKT_SYNC,
	DCCP_PKT_SYNCACK,
	DCCP_PKT_INVALID,
};

#define DCCP_NR_PKT_TYPES DCCP_PKT_INVALID

static inline unsigned int dccp_packet_hdr_len(const __u8 type)
{
	if (type == DCCP_PKT_DATA)
		return 0;
	if (type == DCCP_PKT_DATAACK	||
	    type == DCCP_PKT_ACK	||
	    type == DCCP_PKT_SYNC	||
	    type == DCCP_PKT_SYNCACK	||
	    type == DCCP_PKT_CLOSE	||
	    type == DCCP_PKT_CLOSEREQ)
		return sizeof(struct dccp_hdr_ack_bits);
	if (type == DCCP_PKT_REQUEST)
		return sizeof(struct dccp_hdr_request);
	if (type == DCCP_PKT_RESPONSE)
		return sizeof(struct dccp_hdr_response);
	return sizeof(struct dccp_hdr_reset);
}
enum dccp_reset_codes {
	DCCP_RESET_CODE_UNSPECIFIED = 0,
	DCCP_RESET_CODE_CLOSED,
	DCCP_RESET_CODE_ABORTED,
	DCCP_RESET_CODE_NO_CONNECTION,
	DCCP_RESET_CODE_PACKET_ERROR,
	DCCP_RESET_CODE_OPTION_ERROR,
	DCCP_RESET_CODE_MANDATORY_ERROR,
	DCCP_RESET_CODE_CONNECTION_REFUSED,
	DCCP_RESET_CODE_BAD_SERVICE_CODE,
	DCCP_RESET_CODE_TOO_BUSY,
	DCCP_RESET_CODE_BAD_INIT_COOKIE,
	DCCP_RESET_CODE_AGGRESSION_PENALTY,
};

/* DCCP options */
enum {
	DCCPO_PADDING = 0,
	DCCPO_MANDATORY = 1,
	DCCPO_MIN_RESERVED = 3,
	DCCPO_MAX_RESERVED = 31,
	DCCPO_NDP_COUNT = 37,
	DCCPO_ACK_VECTOR_0 = 38,
	DCCPO_ACK_VECTOR_1 = 39,
	DCCPO_TIMESTAMP = 41,
	DCCPO_TIMESTAMP_ECHO = 42,
	DCCPO_ELAPSED_TIME = 43,
	DCCPO_MAX = 45,
	DCCPO_MIN_CCID_SPECIFIC = 128,
	DCCPO_MAX_CCID_SPECIFIC = 255,
};

/* DCCP features */
enum {
	DCCPF_RESERVED = 0,
	DCCPF_SEQUENCE_WINDOW = 3,
	DCCPF_SEND_ACK_VECTOR = 6,
	DCCPF_SEND_NDP_COUNT = 7,
	/* 10-127 reserved */
	DCCPF_MIN_CCID_SPECIFIC = 128,
	DCCPF_MAX_CCID_SPECIFIC = 255,
};

#ifdef __KERNEL__

#include <linux/in.h>
#include <linux/list.h>
#include <linux/uio.h>
#include <linux/workqueue.h>

#include <net/inet_connection_sock.h>
#include <net/inet_timewait_sock.h>
#include <net/sock.h>
#include <net/tcp_states.h>
#include <net/tcp.h>

enum dccp_state {
	DCCP_OPEN	= TCP_ESTABLISHED,
	DCCP_REQUESTING	= TCP_SYN_SENT,
	DCCP_PARTOPEN	= TCP_FIN_WAIT1, /* FIXME:
					    This mapping is horrible, but TCP has
					    no matching state for DCCP_PARTOPEN,
					    as TCP_SYN_RECV is already used by
					    DCCP_RESPOND, why don't stop using TCP
					    mapping of states? OK, now we don't use
					    sk_stream_sendmsg anymore, so doesn't
					    seem to exist any reason for us to
					    do the TCP mapping here */
	DCCP_LISTEN	= TCP_LISTEN,
	DCCP_RESPOND	= TCP_SYN_RECV,
	DCCP_CLOSING	= TCP_CLOSING,
	DCCP_TIME_WAIT	= TCP_TIME_WAIT,
	DCCP_CLOSED	= TCP_CLOSE,
	DCCP_MAX_STATES = TCP_MAX_STATES,
};

#define DCCP_STATE_MASK 0xf
#define DCCP_ACTION_FIN (1<<7)

enum {
	DCCPF_OPEN	 = TCPF_ESTABLISHED,
	DCCPF_REQUESTING = TCPF_SYN_SENT,
	DCCPF_PARTOPEN	 = TCPF_FIN_WAIT1,
	DCCPF_LISTEN	 = TCPF_LISTEN,
	DCCPF_RESPOND	 = TCPF_SYN_RECV,
	DCCPF_CLOSING	 = TCPF_CLOSING,
	DCCPF_TIME_WAIT	 = TCPF_TIME_WAIT,
	DCCPF_CLOSED	 = TCPF_CLOSE,
};

static inline struct dccp_hdr *dccp_hdr(const struct sk_buff *skb)
{
	return (struct dccp_hdr *)skb->h.raw;
}

static inline struct dccp_hdr_ext *dccp_hdrx(const struct sk_buff *skb)
{
	return (struct dccp_hdr_ext *)(skb->h.raw + sizeof(struct dccp_hdr));
}

static inline unsigned int __dccp_basic_hdr_len(const struct dccp_hdr *dh)
{
	return sizeof(*dh) + (dh->dccph_x ? sizeof(struct dccp_hdr_ext) : 0);
}

static inline unsigned int dccp_basic_hdr_len(const struct sk_buff *skb)
{
	const struct dccp_hdr *dh = dccp_hdr(skb);
	return __dccp_basic_hdr_len(dh);
}

static inline __u64 dccp_hdr_seq(const struct sk_buff *skb)
{
	const struct dccp_hdr *dh = dccp_hdr(skb);
#if defined(__LITTLE_ENDIAN_BITFIELD)
	__u64 seq_nr = ntohl(dh->dccph_seq << 8);
#elif defined(__BIG_ENDIAN_BITFIELD)
	__u64 seq_nr = ntohl(dh->dccph_seq);
#else
#error  "Adjust your <asm/byteorder.h> defines"
#endif

	if (dh->dccph_x != 0)
		seq_nr = (seq_nr << 32) + ntohl(dccp_hdrx(skb)->dccph_seq_low);

	return seq_nr;
}

static inline struct dccp_hdr_request *dccp_hdr_request(struct sk_buff *skb)
{
	return (struct dccp_hdr_request *)(skb->h.raw + dccp_basic_hdr_len(skb));
}

static inline struct dccp_hdr_ack_bits *dccp_hdr_ack_bits(const struct sk_buff *skb)
{
	return (struct dccp_hdr_ack_bits *)(skb->h.raw + dccp_basic_hdr_len(skb));
}

static inline u64 dccp_hdr_ack_seq(const struct sk_buff *skb)
{
	const struct dccp_hdr_ack_bits *dhack = dccp_hdr_ack_bits(skb);
#if defined(__LITTLE_ENDIAN_BITFIELD)
	return (((u64)ntohl(dhack->dccph_ack_nr_high << 8)) << 32) + ntohl(dhack->dccph_ack_nr_low);
#elif defined(__BIG_ENDIAN_BITFIELD)
	return (((u64)ntohl(dhack->dccph_ack_nr_high)) << 32) + ntohl(dhack->dccph_ack_nr_low);
#else
#error  "Adjust your <asm/byteorder.h> defines"
#endif
}

static inline struct dccp_hdr_response *dccp_hdr_response(struct sk_buff *skb)
{
	return (struct dccp_hdr_response *)(skb->h.raw + dccp_basic_hdr_len(skb));
}

static inline struct dccp_hdr_reset *dccp_hdr_reset(struct sk_buff *skb)
{
	return (struct dccp_hdr_reset *)(skb->h.raw + dccp_basic_hdr_len(skb));
}

static inline unsigned int __dccp_hdr_len(const struct dccp_hdr *dh)
{
	return __dccp_basic_hdr_len(dh) +
	       dccp_packet_hdr_len(dh->dccph_type);
}

static inline unsigned int dccp_hdr_len(const struct sk_buff *skb)
{
	return __dccp_hdr_len(dccp_hdr(skb));
}


/* initial values for each feature */
#define DCCPF_INITIAL_SEQUENCE_WINDOW		100
/* FIXME: for now we're using CCID 3 (TFRC) */
#define DCCPF_INITIAL_CCID			3
#define DCCPF_INITIAL_SEND_ACK_VECTOR		0
/* FIXME: for now we're default to 1 but it should really be 0 */
#define DCCPF_INITIAL_SEND_NDP_COUNT		1

#define DCCP_NDP_LIMIT 0xFFFFFF

/**
  * struct dccp_options - option values for a DCCP connection
  *	@dccpo_sequence_window - Sequence Window Feature (section 7.5.2)
  *	@dccpo_ccid - Congestion Control Id (CCID) (section 10)
  *	@dccpo_send_ack_vector - Send Ack Vector Feature (section 11.5)
  *	@dccpo_send_ndp_count - Send NDP Count Feature (7.7.2)
  */
struct dccp_options {
	__u64	dccpo_sequence_window;
	__u8	dccpo_ccid;
	__u8	dccpo_send_ack_vector;
	__u8	dccpo_send_ndp_count;
};

extern void __dccp_options_init(struct dccp_options *dccpo);
extern void dccp_options_init(struct dccp_options *dccpo);
extern int dccp_parse_options(struct sock *sk, struct sk_buff *skb);

struct dccp_request_sock {
	struct inet_request_sock dreq_inet_rsk;
	__u64			 dreq_iss;
	__u64			 dreq_isr;
	__u32			 dreq_service;
};

static inline struct dccp_request_sock *dccp_rsk(const struct request_sock *req)
{
	return (struct dccp_request_sock *)req;
}

extern struct inet_timewait_death_row dccp_death_row;

/* Read about the ECN nonce to see why it is 253 */
#define DCCP_MAX_ACK_VECTOR_LEN 253

struct dccp_options_received {
	u32	dccpor_ndp:24,
		dccpor_ack_vector_len:8;
	u32	dccpor_ack_vector_idx:10;
	/* 22 bits hole, try to pack */
	u32	dccpor_timestamp;
	u32	dccpor_timestamp_echo;
	u32	dccpor_elapsed_time;
};

struct ccid;

enum dccp_role {
	DCCP_ROLE_UNDEFINED,
	DCCP_ROLE_LISTEN,
	DCCP_ROLE_CLIENT,
	DCCP_ROLE_SERVER,
};

/**
 * struct dccp_sock - DCCP socket state
 *
 * @dccps_swl - sequence number window low
 * @dccps_swh - sequence number window high
 * @dccps_awl - acknowledgement number window low
 * @dccps_awh - acknowledgement number window high
 * @dccps_iss - initial sequence number sent
 * @dccps_isr - initial sequence number received
 * @dccps_osr - first OPEN sequence number received
 * @dccps_gss - greatest sequence number sent
 * @dccps_gsr - greatest valid sequence number received
 * @dccps_gar - greatest valid ack number received on a non-Sync; initialized to %dccps_iss
 * @dccps_timestamp_time - time of latest TIMESTAMP option
 * @dccps_timestamp_echo - latest timestamp received on a TIMESTAMP option
 * @dccps_ext_header_len - network protocol overhead (IP/IPv6 options)
 * @dccps_pmtu_cookie - Last pmtu seen by socket
 * @dccps_avg_packet_size - FIXME: has to be set by the app thru some setsockopt or ioctl, CCID3 uses it
 * @dccps_role - Role of this sock, one of %dccp_role
 * @dccps_ndp_count - number of Non Data Packets since last data packet
 * @dccps_hc_rx_ackpkts - receiver half connection acked packets
 */
struct dccp_sock {
	/* inet_connection_sock has to be the first member of dccp_sock */
	struct inet_connection_sock	dccps_inet_connection;
	__u64				dccps_swl;
	__u64				dccps_swh;
	__u64				dccps_awl;
	__u64				dccps_awh;
	__u64				dccps_iss;
	__u64				dccps_isr;
	__u64				dccps_osr;
	__u64				dccps_gss;
	__u64				dccps_gsr;
	__u64				dccps_gar;
	unsigned long			dccps_service;
	unsigned long			dccps_timestamp_time;
	__u32				dccps_timestamp_echo;
	__u32				dccps_avg_packet_size;
	unsigned long			dccps_ndp_count;
	__u16				dccps_ext_header_len;
	__u32				dccps_pmtu_cookie;
	__u32				dccps_mss_cache;
	struct dccp_options		dccps_options;
	struct dccp_ackpkts		*dccps_hc_rx_ackpkts;
	void				*dccps_hc_rx_ccid_private;
	void				*dccps_hc_tx_ccid_private;
	struct ccid			*dccps_hc_rx_ccid;
	struct ccid			*dccps_hc_tx_ccid;
	struct dccp_options_received	dccps_options_received;
	enum dccp_role			dccps_role:2;
};
 
static inline struct dccp_sock *dccp_sk(const struct sock *sk)
{
	return (struct dccp_sock *)sk;
}

static inline const char *dccp_role(const struct sock *sk)
{
	switch (dccp_sk(sk)->dccps_role) {
	case DCCP_ROLE_UNDEFINED: return "undefined";
	case DCCP_ROLE_LISTEN:	  return "listen";
	case DCCP_ROLE_SERVER:	  return "server";
	case DCCP_ROLE_CLIENT:	  return "client";
	}
	return NULL;
}

#endif /* __KERNEL__ */

#endif /* _LINUX_DCCP_H */
